package org.eso.phase3.validator;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.log4j.Logger;
import org.eso.util.misc.StreamConsumer;
import org.eso.util.misc.ThreadWatchdog;

/**
 * This class is a copy of {@link org.eso.util.misc.SystemCommand} but with the
 * String commandLine replaced with a String[]. This allows for spaces embedded
 * in the command line component.
 */
public class SystemCommandFromArray
{
    /** Apache Log4J logger. */
    static final Logger logger = Logger.getLogger(SystemCommandFromArray.class);

    /** The name of this class for logging purposes. */
    private static final String classLogName = "SystemCommandFromArray";

    /** The command line to execute. */
    private volatile String[] commandLine = null;

    /**
     * Binary semaphore, used to ensure that either {@link #execute()} is
     * updating the results of a command execution, or the results are being
     * read by an accessor method, but not both at once.
     */
    private final Semaphore commandResultPermit = new Semaphore(0);

    /** The command's exit status. */
    private int exitStatus = 0;

    /** An interrupt used for timeouts and/or explicitly stopping the command. */
    private volatile ThreadWatchdog.Interrupt interrupt = null;

    /** True if the command is currently executing. */
    private volatile boolean isExecuting = false;

    /** True if this SystemCommandFromArray has never executed a command before. */
    private boolean newSystemCommand = true;

    /** A StreamConsumer to collect the command's standard error output. */
    private final StreamConsumer standardErrorConsumer;

    /** A string storing the command's standard error output. */
    private String standardErrorString = "";

    /** A StreamConsumer to collect the command's standard output. */
    private final StreamConsumer standardOutputConsumer;

    /** A string storing the command's standard output. */
    private String standardOutputString = "";

    /**
     * The timeout this command should be granted before it is automatically
     * terminated. Specified in seconds. If this is zero, no timeout will be
     * applied and automatic termination will be disabled.
     */
    private volatile long timeout = 0;

    /** True if the timeout is counting down. */
    private volatile boolean timeoutCounting = false;

    /**
     * The working directory this command should have. If this is null, the
     * command will be executed in the current process' working directory.
     */
    private volatile File workingDirectory = null;

    /**
     * Constructs a new SystemCommandFromArray object whose command line will be
     * specified later. The working directory will default to the current
     * process' working directory. Stream consumption threads will have
     * JVM-generated names.
     */
    public SystemCommandFromArray()
    {
        final String methodLogName = classLogName
                + "::SystemCommandFromArray()";

        logger.trace(methodLogName);

        // Normal output and error stream consumers for the command.
        standardOutputConsumer = new StreamConsumer();
        standardErrorConsumer = new StreamConsumer();
    }

    /**
     * Constructs a new SystemCommandFromArray object to execute the specified
     * command in the specified working directory. Stream consumption threads
     * will have JVM-generated names.
     * 
     * @param workingDirectory
     *            the working directory this command should have. Set this to
     *            null if it should default to the current process' working
     *            directory.
     * @param commandLine
     *            the command line to execute. Set this to the empty string if
     *            it will be specified later.
     * 
     * @throws NullPointerException
     *             if commandLine is null.
     */
    public SystemCommandFromArray(final File workingDirectory,
            final String[] commandLine) throws NullPointerException
    {
        final String methodLogName = classLogName
                + "::SystemCommandFromArray(File, String)";

        logger.trace(methodLogName);

        // Normal output and error stream consumers for the command.
        standardOutputConsumer = new StreamConsumer();
        standardErrorConsumer = new StreamConsumer();

        // Set the command line and working directory.
        setCommandLine(commandLine);
        setWorkingDirectory(workingDirectory);
    }

    /**
     * Constructs a new SystemCommandFromArray object to execute the specified
     * command in the specified working directory, terminating it after a given
     * timeout has expired.
     * 
     * @param workingDirectory
     *            the working directory this command should have. Set this to
     *            null if it should default to the current process' working
     *            directory.
     * @param commandLine
     *            the command line to execute. Set this to the empty string if
     *            it will be specified later.
     * @param timeout
     *            the timeout this command should be granted before it is
     *            automatically terminated. Specified in seconds. Set this to
     *            zero if automatic termination should be disabled.
     * 
     * @throws IllegalArgumentException
     *             if timeout is negative.
     * @throws NullPointerException
     *             if commandLine is null.
     */
    public SystemCommandFromArray(final File workingDirectory,
            final String[] commandLine, final long timeout)
            throws IllegalArgumentException, NullPointerException
    {
        final String methodLogName = classLogName
                + "::SystemCommandFromArray(File, String, String)";

        logger.trace(methodLogName);

        // Normal output and error stream consumers for the command.
        standardOutputConsumer = new StreamConsumer();
        standardErrorConsumer = new StreamConsumer();

        // Set the command line, working directory and timeout.
        setCommandLine(commandLine);
        setWorkingDirectory(workingDirectory);
        setTimeout(timeout);
    }

    /**
     * Constructs a new SystemCommandFromArray object to execute the specified
     * command in the specified working directory, terminating it after a given
     * timeout has expired. In addition, the stream consumption threads will be
     * given names with the specified prefix.
     * 
     * @param workingDirectory
     *            the working directory this command should have. Set this to
     *            null if it should default to the current process' working
     *            directory.
     * @param commandLine
     *            the command line to execute. Set this to the empty string if
     *            it will be specified later.
     * @param timeout
     *            the timeout this command should be granted before it is
     *            automatically terminated. Specified in seconds. Set this to
     *            zero if automatic termination should be disabled.
     * @param threadNamePrefix
     *            a prefix to use in stream consumption thread names. The two
     *            threads will be named "<i>threadNamePrefix</i>[STDOUT]" and
     *            "<i>threadNamePrefix</i>[STDERR]" for the command's normal
     *            output and error streams, respectively.
     * 
     * @throws IllegalArgumentException
     *             if timeout is negative.
     * @throws NullPointerException
     *             if commandLine or threadNamePrefix is null.
     */
    public SystemCommandFromArray(final File workingDirectory,
            final String[] commandLine, final long timeout,
            final String threadNamePrefix) throws IllegalArgumentException,
            NullPointerException
    {
        final String methodLogName = classLogName
                + "::SystemCommandFromArray(File, String, String)";

        logger.trace(methodLogName);

        if (threadNamePrefix == null)
        {
            final String message = methodLogName
                    + " - threadNamePrefix must not be null.";

            logger.fatal(message);
            throw new NullPointerException(message);
        }

        // Normal output and error stream consumers for the command.
        standardOutputConsumer = new StreamConsumer(threadNamePrefix
                + "[STDOUT]");
        standardErrorConsumer = new StreamConsumer(threadNamePrefix
                + "[STDERR]");

        // Set the command line, working directory and timeout.
        setCommandLine(commandLine);
        setWorkingDirectory(workingDirectory);
        setTimeout(timeout);
    }

    /**
     * Constructs a new SystemCommandFromArray object to execute the specified
     * command. The working directory will default to the current process'
     * working directory. Stream consumption threads will have JVM-generated
     * names.
     * 
     * @param commandLine
     *            the command line to execute. Set this to the empty string if
     *            it will be specified later.
     * 
     * @throws NullPointerException
     *             if commandLine is null.
     */
    public SystemCommandFromArray(final String[] commandLine)
            throws NullPointerException
    {
        final String methodLogName = classLogName
                + "::SystemCommandFromArray(String)";

        logger.trace(methodLogName);

        // Normal output and error stream consumers for the command.
        standardOutputConsumer = new StreamConsumer();
        standardErrorConsumer = new StreamConsumer();

        // Set the command line.
        setCommandLine(commandLine);
    }

    /**
     * Blocks until the current command execution is complete. If no command has
     * been executed yet, this method will block until a command has been
     * executed and command execution is complete.
     */
    public void awaitCompletion()
    {
        final String methodLogName = classLogName + "::awaitCompletion()";

        logger.trace(methodLogName);

        // Wait for the command execution results to become available.
        commandResultPermit.acquireUninterruptibly();

        // As a binary semaphore, the commandResultPermit must never have
        // any permits available at this stage.
        if (commandResultPermit.availablePermits() != 0)
        {
            final String message = methodLogName + " - [commandResultPermit]"
                    + " binary semaphore violation detected.";

            logger.fatal(message);
            throw new AssertionError(message);
        }

        // Release the command execution results.
        commandResultPermit.release();
    }

    /**
     * <p>
     * Executes the command, waits for it to finish executing, then returns its
     * exit status.
     * </p>
     * 
     * <p>
     * NOTE: this method does not terminate the command if it receives an
     * unexpected interrupt. Instead, the command runs to completion, and the
     * thread's interrupt flag is reinstated after the command returns. The
     * command can be terminated immediately by calling {@link #interrupt()}
     * (this causes execute() to throw an InterruptedException).
     * </p>
     * 
     * @throws IllegalStateException
     *             if the command line to execute has not been set yet.
     * @throws InterruptedException
     *             if command execution was explicitly interrupted by calling
     *             {@link #interrupt()}. This is the only case where execute()
     *             throws an InterruptedException.
     * @throws IOException
     *             if an I/O error occurs whilst attempting to commence
     *             execution.
     * @throws TimeoutException
     *             if a timeout has been set for the command and command
     *             execution takes too long.
     * 
     * @return an integer exit status for the command.
     */
    public synchronized int execute() throws IllegalStateException,
            InterruptedException, IOException, TimeoutException
    {
        final String methodLogName = classLogName + "::execute()";

        logger.trace(methodLogName);

        if (commandLine == null)
        {
            final String message = methodLogName
                    + " - command line has not been set yet.";

            logger.fatal(message);
            throw new IllegalStateException(message);
        }

        // Lock out any attempts to get the results of a previous command
        // execution before starting the new command execution.
        if (newSystemCommand)
        {
            // This is a special case; when executing a command for the
            // first time, no previous results have ever been released.
            logger.debug(methodLogName
                    + " - this is a new SystemCommandFromArray.");
            newSystemCommand = false;
        }
        else
        {
            // This is the normal case; acquire the permit.
            commandResultPermit.acquireUninterruptibly();
        }

        // As a binary semaphore, the commandResultPermit must never have
        // any permits available at this stage.
        if (commandResultPermit.availablePermits() != 0)
        {
            final String message = methodLogName + " - [commandResultPermit]"
                    + " binary semaphore violation detected.";

            logger.fatal(message);
            throw new AssertionError(message);
        }

        // Mark execution as being in progress.
        isExecuting = true;

        // Reset all command results.
        standardOutputString = "";
        standardErrorString = "";
        exitStatus = -1;

        // Prepare to execute the command.
        final Runtime runtime = Runtime.getRuntime();
        Process process = null;

        logger.debug(methodLogName
                + " - preparing to execute command ["
                + Arrays.toString(commandLine)
                + "] in working directory ["
                + workingDirectory
                + (timeout > 0 ? "] with a " + timeout + " second timeout."
                        : "]."));

        // Execute the command.
        try
        {
            process = runtime.exec(commandLine, null, workingDirectory);
        }
        catch( final IOException ioe )
        {
            logger.error(methodLogName + " - I/O error [" + ioe.getMessage()
                    + "] whilst attempting to execute command.");

            // Command execution has failed completely.
            // 
            // Release the (blank) command results to other methods.
            commandResultPermit.release();

            // Rethrow the exception.
            throw ioe;
        }

        // Command execution is now in progress.
        logger.debug(methodLogName + " - command execution in progress.");

        // Consume the command's output streams.
        standardOutputConsumer.consumeStream(process.getInputStream());
        standardErrorConsumer.consumeStream(process.getErrorStream());

        // True if this thread has been unexpectedly interrupted.
        boolean unexpectedInterrupt = Thread.interrupted();

        if (timeout > 0)
        {
            // Timeouts are enabled - register this thread with the
            // watchdog for an automatic interrupt.
            interrupt = ThreadWatchdog.register(Thread.currentThread(),
                    timeout, TimeUnit.SECONDS);
            timeoutCounting = true;
        }
        else
        {
            // Timeouts are disabled - register this thread with the
            // watchdog, so the command can be explicitly interrupted,
            // but without scheduling an automatic interrupt.
            interrupt = ThreadWatchdog.register(Thread.currentThread());
            timeoutCounting = false;
        }

        // Wait for the command to terminate, storing its exit status
        // when it does.
        while( true )
        {
            try
            {
                exitStatus = process.waitFor();

                // Cancel the interrupt.
                interrupt.cancel();
            }
            catch( final InterruptedException ie )
            {
                if (interrupt.executed())
                {
                    // The command has been interrupted - terminate it.
                    process.destroy();

                    // If an unexpected interrupt was detected, set this
                    // thread's interrupt status.
                    if (unexpectedInterrupt)
                    {
                        Thread.currentThread().interrupt();
                    }

                    // Store the command's results, if any, and release them
                    // to other methods.
                    standardOutputString = standardOutputConsumer.getConsumedString();
                    standardErrorString = standardErrorConsumer.getConsumedString();

                    commandResultPermit.release();

                    // The command has been terminated.
                    isExecuting = false;

                    if (timeoutCounting)
                    {
                        final String logMessage = methodLogName
                                + " - command timeout exceeded after "
                                + timeout
                                + (timeout == 1 ? " second.": " seconds.")
                                + " Command terminated.";

                        logger.error(logMessage);

                        throw new TimeoutException(logMessage);
                    }
                    else
                    {
                        final String logMessage = methodLogName
                                + " - command explicitly interrupted.";

                        logger.debug(logMessage);

                        throw new InterruptedException(logMessage);
                    }
                }
                else
                {
                    // This thread was unexpectedly interrupted - flag it
                    // and keep waiting for command execution to finish.
                    logger.debug(methodLogName
                            + " - unexpected interrupt received whilst"
                            + " waiting for command execution to finish.");

                    unexpectedInterrupt = true;
                    continue;
                }
            }

            // The command has stopped executing, one way or another,
            // so the interrupt is no longer required.
            interrupt = null;
            break;
        }

        // If an unexpected interrupt was detected, set this thread's
        // interrupt status.
        if (unexpectedInterrupt)
        {
            Thread.currentThread().interrupt();
        }

        // Store the command's results and release them to other methods.
        standardOutputString = standardOutputConsumer.getConsumedString();
        standardErrorString = standardErrorConsumer.getConsumedString();
        commandResultPermit.release();

        // Command execution is complete.
        logger.debug(methodLogName + " - command execution complete.");
        isExecuting = false;

        // Explicitly destroy the process in order to release any system
        // resources still being held for it [this fixes issue DFS05501].
        process.destroy();

        // Return the command's exit status.
        return (exitStatus);
    }

    /**
     * Returns the command line this SystemCommandFromArray will execute, or
     * null if no command line has been set.
     */
    public String[] getCommandLine()
    {
        final String methodLogName = classLogName + "::getCommandLine()";

        logger.trace(methodLogName);

        return (commandLine);
    }

    /**
     * Returns the exit status of the last command execution. If no command has
     * been executed yet, or a command execution is still in progress, this
     * method will block until the command has been executed.
     */
    public int getExitStatus()
    {
        final String methodLogName = classLogName + "::getExitStatus()";

        logger.trace(methodLogName);

        // Wait for the command execution results to become available.
        commandResultPermit.acquireUninterruptibly();

        // As a binary semaphore, the commandResultPermit must never have
        // any permits available at this stage.
        if (commandResultPermit.availablePermits() != 0)
        {
            final String message = methodLogName + " - [commandResultPermit]"
                    + " binary semaphore violation detected.";

            logger.fatal(message);
            throw new AssertionError(message);
        }

        // Get the exit status.
        final int result = exitStatus;

        // Release the command execution results and return the exit status.
        commandResultPermit.release();
        return (result);
    }

    /**
     * Returns everything the last command execution wrote to its standard error
     * stream. If no command has been executed yet, or a command execution is
     * still in progress, this method will block until the command has been
     * executed.
     */
    public String getStandardErrorString()
    {
        final String methodLogName = classLogName
                + "::getStandardErrorString()";

        logger.trace(methodLogName);

        // Wait for the command execution results to become available.
        commandResultPermit.acquireUninterruptibly();

        // As a binary semaphore, the commandResultPermit must never have
        // any permits available at this stage.
        if (commandResultPermit.availablePermits() != 0)
        {
            final String message = methodLogName + " - [commandResultPermit]"
                    + " binary semaphore violation detected.";

            logger.fatal(message);
            throw new AssertionError(message);
        }

        // Get the standard error output string.
        final String result = standardErrorString;

        // Release the command execution results and return the string.
        commandResultPermit.release();
        return (result);
    }

    /**
     * Returns everything the last command execution wrote to its standard
     * output stream. If no command has been executed yet, or a command
     * execution is still in progress, this method will block until the command
     * has been executed.
     */
    public String getStandardOutputString()
    {
        final String methodLogName = classLogName
                + "::getStandardOutputString()";

        logger.trace(methodLogName);

        // Wait for the command execution results to become available.
        commandResultPermit.acquireUninterruptibly();

        // As a binary semaphore, the commandResultPermit must never have
        // any permits available at this stage.
        if (commandResultPermit.availablePermits() != 0)
        {
            final String message = methodLogName + " - [commandResultPermit]"
                    + " binary semaphore violation detected.";

            logger.fatal(message);
            throw new AssertionError(message);
        }

        // Get the standard output string.
        final String result = standardOutputString;

        // Release the command execution results and return the string.
        commandResultPermit.release();
        return (result);
    }

    /**
     * Returns the timeout this SystemCommandFromArray will apply to its
     * command, or zero if automatic termination is disabled.
     */
    public long getTimeout()
    {
        final String methodLogName = classLogName + "::getTimeout()";

        logger.trace(methodLogName);

        return (timeout);
    }

    /**
     * Returns the working directory this SystemCommandFromArray will execute
     * in, or null if the command will be executed in the current process'
     * working directory.
     */
    public File getWorkingDirectory()
    {
        final String methodLogName = classLogName + "::getWorkingDirectory()";

        logger.trace(methodLogName);

        return (workingDirectory);
    }

    /**
     * Tries to interrupt the command on a best-effort basis, if it is currently
     * executing. This method is not guaranteed to work, but can be retried if
     * necessary. See {@link #isExecuting()}.
     */
    public void interrupt()
    {
        final String methodLogName = classLogName + "::isExecuting()";

        logger.trace(methodLogName);

        // Take a private copy - the interrupt field is volatile and this
        // method is asynchronous, so the field could change at any time.
        final ThreadWatchdog.Interrupt interrupt = this.interrupt;

        if (interrupt != null)
        {
            // Cancel any timeout and execute the interrupt.
            timeoutCounting = false;
            interrupt.execute();
        }
    }

    /**
     * Returns true if the command is currently executing, false otherwise.
     */
    public boolean isExecuting()
    {
        final String methodLogName = classLogName + "::isExecuting()";

        logger.trace(methodLogName);

        return (isExecuting);
    }

    /**
     * Sets the command line this SystemCommandFromArray will execute. If a
     * command is currently executing, this method blocks until execution is
     * complete, then sets the command line.
     * 
     * @param commandLine
     *            the command line to execute.
     * 
     * @throws NullPointerException
     *             if commandLine is null.
     */
    public synchronized void setCommandLine(final String[] commandLine)
            throws NullPointerException
    {
        final String methodLogName = classLogName + "::setCommandLine()";

        logger.trace(methodLogName);

        if (commandLine == null)
        {
            final String message = methodLogName
                    + " - commandLine must not be null.";

            logger.fatal(message);
            throw new NullPointerException(message);
        }

        // Set the command line.
        this.commandLine = Arrays.copyOf(commandLine, commandLine.length);
    }

    /**
     * Sets the timeout that this SystemCommandFromArray will apply to its
     * command. If a command is currently executing, this method blocks until
     * execution is complete, then sets the timeout.
     * 
     * @param timeout
     *            the timeout this command should have. Specified in seconds.
     *            Set this to zero if automatic termination should be disabled.
     * 
     * @throws IllegalArgumentException
     *             if timeout is negative.
     */
    public synchronized void setTimeout(final long timeout)
            throws IllegalArgumentException
    {
        final String methodLogName = classLogName + "::setTimeout()";

        logger.trace(methodLogName);

        if (timeout < 0)
        {
            final String message = methodLogName + " - timeout [" + timeout
                    + "] must not be negative.";

            logger.fatal(message);
            throw new IllegalArgumentException(message);
        }

        // Set the timeout.
        this.timeout = timeout;
    }

    /**
     * Sets the working directory this SystemCommandFromArray will execute in.
     * If a command is currently executing, this method blocks until execution
     * is complete, then sets the working directory.
     * 
     * @param workingDirectory
     *            the working directory this command should have. Set this to
     *            null if it should default to the current process' working
     *            directory.
     */
    public synchronized void setWorkingDirectory(final File workingDirectory)
    {
        final String methodLogName = classLogName + "::setWorkingDirectory()";

        logger.trace(methodLogName);

        // Set the working directory.
        this.workingDirectory = workingDirectory;
    }
}
