package org.eso.phase3.validator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * This class encapsulates the results provided by the Validator application of
 * all the performed checks on a phase 3 package. ValidationReport objects can 
 * be organized in trees (parent/child relationship). When the tree is build the 
 * statuses of the reports are synchronized, i.e. if a child is in error, the 
 * parent goes in error as well.  
 * 
 * @author dsforna
 * 
 */
public class ValidationReport
{

    /**
     * Status of this report is one of these enums. <b>WARNING:</b> the order of
     * the defined enumeration values is important, so I can take advantages of
     * the complier-generated ordinal() and compareTo() to merge the status of
     * two reports.
     * 
     * @author dsforna
     * 
     */
    public enum STATUS
    {
        VALID, WARN,ERROR, FATAL;
    };

    /**
     * Each message of this report has an associated level.
     * @author dsforna
     */
    private enum MSGLEVEL
    {
        ERROR, FATAL, INFO, WARN;
    };

    
    /**
     * A container to internally keep together messages and associated levels.
     * @author dsforna
     */
    static class MsgAndLevel
    {
        final MSGLEVEL level;
        final String msg;
        MsgAndLevel(final String msg, final MSGLEVEL level)
        {
            this.msg = msg;
            this.level = level;
        }
    };

    public static String newline = Consts.NEWLINE;

    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidationReport.class);

    /**
     * indentation between the messages of a report and the messages of its
     * direct sub-reports.
     */
    private static final String padString = " ";

    private final String description;

    private final List<MsgAndLevel> messages = new ArrayList<MsgAndLevel>();

    private ValidationReport parent = null;

    private STATUS status;

    private final List<ValidationReport> subReport;

    public ValidationReport(final String description,
            final List<ValidationReport> subreport)
    {
        this.description = description;
        this.subReport = new ArrayList<ValidationReport>();
        status = STATUS.VALID;
        if (subreport != null)
        {
            for (final ValidationReport r : subreport)
            {
                addsubreport(r);
            }
        }
    }

    /**Add the input subreport as child of this report. 
     * Note that {@link #setParent(ValidationReport)} method of the subreport 
     * is automatically called by this method and the status of this report is 
     * adapted to the status of the added child.
     * @param r the input subreport.
     */
    public void addsubreport(final ValidationReport r)
    {
        r.setParent(this);
        subReport.add(r);
        attemptStatus(r.getStatus());
    }


    public void addError(final String msg)
    {
        addmsg(msg, MSGLEVEL.ERROR);
    }

    public void addFatal(final String msg)
    {
        addmsg(msg, MSGLEVEL.FATAL);
    }

    public void addInfo(final String msg)
    {
        addmsg(msg, MSGLEVEL.INFO);
    }

    public void addWarn(final String msg)
    {
        addmsg(msg, MSGLEVEL.WARN);
    }

    /**
     * Attempt to set this status to the input status. It succeeds only if the
     * input value is a status "worse" of the current status. With this method
     * the current status of this object can change from VALID to WARN to ERROR,
     * but not from ERROR back to VALID, only with <code>setStatus</code> it can
     * go back from ERROR to VALID.
     * 
     * @param status
     *            the input status
     */
    public void attemptStatus(final STATUS status)
    {
        if (status.compareTo(this.status) > 0)
        {
            logger.debug("Changing status from " + this.status.toString()
                    + " to " + status.toString() + " in report: " + description);
            setStatus(status);
        }
        else if (status.compareTo(this.status) == 0)
        {
            logger.trace("Nothing to do: report is already in status ="
                    + status.toString());
        }
        else
        {
            logger.debug("Not changing status from " + this.status.toString()
                    + " to " + status.toString() + " in report: " + description
                    + " because current status is worse than input status.");
        }
    }

    /**
     * Add a message to the list corresponding to the input status and then call
     * {@link #attemptStatus(STATUS status)}.
     * 
     * @param status
     *            the input status.
     * @param msg
     *            the message to add to the list of message for status.
     */
    public void attemptStatus(final STATUS status, final String msg)
    {
        attemptStatus(status);
        if ((msg == null) || msg.equals(""))
        {
            return;
        }
        switch( status ) {
            case VALID:
                addInfo(msg);
            break;
            case WARN:
                addWarn(msg);
            break;
            case ERROR:
                addError(msg);
            break;
            case FATAL:
                addFatal(msg);
            break;
            default:
                logger.error("unknown message categoryMap for: " + msg);
        }

    }

    public String getDescription()
    {
        return description;
    }

    /**
     * @return The error messages in this report and all the sub-reports. Or the
     *         empty string if there are no errors.
     */
    public String getError()
    {
        return getMsgsWithLevel(MSGLEVEL.ERROR);    
    }

    /**
     * @return The warning messages in this report and all the sub-reports. Or the
     *         empty string if there are no warning.
     */
    public String getWarning()
    {
        return getMsgsWithLevel(MSGLEVEL.WARN);    
    }
    
    
    /**
     * @return The messages with the level given in input, in this report and 
     * all the sub-reports. Or the empty string if there are no messages with 
     * the level given in input.
     */
    private String getMsgsWithLevel(MSGLEVEL inputLevel)
    {
        String tmp = "";

        for (int index = 0; index < messages.size(); index++)
        {
            if (messages.get(index).level == inputLevel)
            {
                tmp += messages.get(index).msg;
            }
            if (!tmp.equals(""))
            {
                if (!tmp.endsWith(newline))
                {
                    tmp += newline;
                }
            }
        }
        for (final ValidationReport sv : subReport)
        {
            tmp += sv.getMsgsWithLevel(inputLevel);
        }
        return tmp;
    }
    
 
    /**
     * @return the status of this report. Note that this status is always in 
     * synch with the status of nay subreport (for instance it cannot be VALID 
     * if any subreport is in ERROR).
     */
    public STATUS getStatus()
    {
        return status;
    }

    /**
     * Set the parent of this report. Do not adapt the status of the parent to 
     * the status of this report (this is done by {@link #addsubreport(ValidationReport)}).
     * 
     * @param parent
     */
    private void setParent(final ValidationReport parent)
    {
        this.parent = parent;
    }

    /** 
     * Setter for the status of this object. The new status is propagated to 
     * the parent.
     * @param status
     */
    public void setStatus(final STATUS status)
    {
        this.status = status;
        if (parent != null)
        {
            parent.attemptStatus(status);
        }
    }

    /**
     * @return a String with the formatted messages of this report.
     */
    @Override
    public String toString()
    {
        return paddedString("");
    }

    /**
     * Format the messages of this report.
     * 
     * @param padding
     *            the initial pad to add to each output line.
     * @return a String with the formatted messages.
     */
    String paddedString(String padding)
    {
        if (padding == null)
        {
            padding = "";
        }

        final String title = description + newline;
        final String footer = padding + "End of " + description + " ."
                + newline;
        String allMsg = title;
        allMsg += padding + "Status: " + getStatus() + newline;
        allMsg += cronFormat(padding);
        final Enumeration<ValidationReport> v = Collections.enumeration(subReport);
        while( v.hasMoreElements() )
        {
            allMsg += padding + "Sub Step: ";
            allMsg += v.nextElement().paddedString(padding + padString);
        }
        allMsg += footer;
        return allMsg;
    }


    private void addmsg(final String msg, final MSGLEVEL level)
    {
        messages.add(new MsgAndLevel(msg, level));
    }

    /**
     * Put in a single string the messages of this report in chronological order 
     * (i.e. not ordered by level).
     * @param padding
     * @return
     */
    private String cronFormat(final String padding)
    {
        String tmp = "";
        for (int index = 0; index < messages.size(); index++)
        {
            tmp += padding + messages.get(index).level.toString() + " - "
                    + messages.get(index).msg + newline;
        }
        return tmp;
    }
}
