package org.eso.phase3.validator;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Appender;
import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.eso.oca.fits.DataTransportFormatHandler;
import org.eso.oca.fits.FITSHandler;
import org.eso.oca.fits.FileHandlerException;
import org.eso.oca.fits.HDRHandler;
import org.eso.oca.fits.TypedHeaderCard;
import org.eso.oca.fits.TypedHeaderCardException;
import org.eso.util.filesystem.RecursiveFileSelectionModel;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/** Collection of utilities for the release validation tool.
 * @author dsforna
 */
public abstract class ValidationUtil
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidationUtil.class);

    /**Files .hdr are or are not accepted as fits files according to this value.*/
    private static boolean acceptHdrFilesAsFits = Consts.DEFAULT_ENABLE_HDR_FILES;

    /**Set the verbose level for the log. Currently the log can only be set 
     * to verbose, but cannot be set back to non-verbose.
     * Setting to verbose means setting all the Logger objects to DEBUG and 
     * all the Appender objects threshold to DEBUG. Note that Appender is 
     * the common interface but it does not have a setThreshold method, the 
     * most upward class in the hierarchy with this method is AppenderSkeleton. 
     * @param verbose true/false if true the log is set to verbose. If false, 
     * the method does nothing.
     */
    public static void setDebugLog(boolean verbose)
    {
        if (! verbose)
        {
            return;
        }
        
        // This is a common threshold for all the loggers.
        LogManager.getLoggerRepository().setThreshold(Level.DEBUG);

        Enumeration<Logger> e = LogManager.getCurrentLoggers();
        List<Logger> allLoggers = new ArrayList<Logger>();
        while( e.hasMoreElements() ) {
            Logger log = e.nextElement();
            allLoggers.add(log);
        }
        allLoggers.add(Logger.getRootLogger());
        
        // TODO: each appender's threshold is set N times, once for each logger 
        // which logging on it. Is there a way to get all the appenders in this 
        // LoggerRepository? 
        for (Logger log: allLoggers) 
        {
            log.setLevel(Level.DEBUG);
            final Enumeration<Appender> ea = log.getAllAppenders();
            while( ea.hasMoreElements() )
            {
                Appender a = ea.nextElement();
                if (a instanceof AppenderSkeleton)
                {
                    if (a instanceof ConsoleAppender) 
                    {
                        // messages on console interfere with the progress bar:
                        continue;
                    }
                    AppenderSkeleton af = (AppenderSkeleton)a;
                    af.setThreshold(Level.DEBUG);
                }
            }
        }

        // Few loggers are a bit too verbose:
        Logger.getLogger("org.eso.oca.parser.SimpleNode").setLevel(Level.WARN);
        // Create a parent logger for all the parser.AST* loggers and set its 
        // level to INFO
        Logger.getLogger("org.eso.oca.parser").setLevel(Level.INFO);
        // This is actually to stop logging of StreamConsumer class:
        Logger.getLogger("org.eso.util.misc").setLevel(Level.INFO);
    }
    
    
    /**
     * Configure Log4j 
     */
    public static void configureLog4j()
    {
    	/* for some reason this does not work if we include the package common
    	 * since this function is currently called only by the main, and no other application does it,
    	 * it's ok to override the log4j configuration.
    	 */
//        boolean rootIsConfigured = !(Logger.getRootLogger().getAllAppenders() instanceof NullEnumeration);
//        if (rootIsConfigured) 
//        {
//            logger.debug("Log4j was already externally configured.");
//            return;
//        } 
        try 
        {
            Properties p = new Properties();
            InputStream is = ClassLoader.getSystemResourceAsStream(
                    Consts.LOG4J_CONF_FILE);
            p.load(is);
            PropertyConfigurator.configure(p);
            is.close();
            logger.debug("Log4j configured from property file: " 
                    + Consts.LOG4J_CONF_FILE);
        } 
        catch (Exception e) 
        {
            BasicConfigurator.configure();
            logger.warn("Error while configuring Log4j: " + e.toString());
            logger.info("Log4j will use the BasicConfigurator.");
        }
     }
    
    
    /**Compute the md5sum of the file with the input path.
     * This method is thread safe because the underlying implementation in the 
     * class {@link org.apache.commons.codec.digest.DigestUtils} is thread safe. 
     * @param filePathName
     * @return
     * @throws IOException if the input path does not correspond to an existing 
     * file or if the actual md5 computation raises an exception.   
     */
    public static String md5sum(String filePathName) throws IOException
    {
        InputStream is = new FileInputStream(new File(filePathName));
        String md5 = DigestUtils.md5Hex(is);
        return md5;
    }
    
    
    /**Set if hdr files are accepted or not.
     * @param val true if .hdr files are accepted, false otherwise.
     */
    public static void setAcceptHdrFilesAsFits(boolean val) 
    {
        acceptHdrFilesAsFits = val;
        if(acceptHdrFilesAsFits)
        {
            logger.debug("Header files are accepted as fits files.");
        } 
        else 
        {
            logger.debug("Header files are not accepted as fits files.");
        }
    }
    
    
    /**Find the index of the header containing the metadata with the release structure. 
     * This can be either the first header (primary header, index =0) for fits 
     * file or fits tile compressed files, or the first extension (index=1) for 
     * some fits tile compressed files. The rule is: if (and only if) with the 
     * compression the primary header is "pushed down" as first extension, then 
     * it contains the keyword ZSIMPLE. 
     * @param fh DataTransportFormatHandler to extract keywords from the fits files.
     * @param filename fits file to parse.
     * @return index of the header to parse.
     * @throws FileHandlerException
     * @throws TypedHeaderCardException
     */
    public static int indexHeaderToParse(DataTransportFormatHandler fh, String filename) 
        throws FileHandlerException, TypedHeaderCardException 
    {
        final String[] req_kwd = new String[] { Consts.ZSIMPLE_KW };
        int primaryHeaderIndex = 0;
        int indexFirstExtension = 1;
        int hduNum = fh.getNumberOfHDUs();
        if (ValidationUtil.isFitsCompressedFile(filename) && (indexFirstExtension < hduNum)) {
            // Check if the first extension has a ZSIMPLE keyword:
            TypedHeaderCard[] cards = fh.getFITSCards(indexFirstExtension, req_kwd);
            if (cards[0].isDefined()) {
                    logger.debug("Found keyword "+ Consts.ZSIMPLE_KW + " in header index="
                            + indexFirstExtension + " of file " + filename);
                    return indexFirstExtension;
            }
        }
        return primaryHeaderIndex;
    }

    
    /**Find the index of the header containing the release structure. This function
     * internally allocates and deallocates its own DataTransportFormatHandler
     * object. Client which already allocates a DataTransportFormatHandler should
     * use  {@link#indexHeaderToParse(DataTransportFormatHandler, String)} instead. 
     * @param fullPathNameOnDisk the full path and name on disk of the fits file 
     * for which the header index is returned. 
     * @return the index of the header with the release meta-data (0 or 1).
     * @throws IOException if fullPathNameOnDisk does not exist or cannot be read.
     * @throws FileHandlerException if the DataTransportFormatHandler object 
     * cannot be created.
     * @throws TypedHeaderCardException if there is an error in parsing the fits 
     * header(s).
     */
    public static int indexHeaderToParse(String fullPathNameOnDisk) 
            throws FileHandlerException, IOException, TypedHeaderCardException
    {
        
        File f = new File(fullPathNameOnDisk);
        if (! f.exists()) 
        { 
            throw new IOException(fullPathNameOnDisk + ": not existing file."); 
        }
        if (! f.canRead()) 
        { 
            throw new IOException(fullPathNameOnDisk + ": cannot read file."); 
        }
        
        String fileName = f.getName();
        DataTransportFormatHandler fh = allocateDTFH(fullPathNameOnDisk);
        int index = indexHeaderToParse(fh, fileName);
        fh.dispose();
        
        return index;
    }
    
    
    /**
     * Extract a Map with entries (String, Set&lt;String&gt;) from a JSON
     * object.
     * 
     * @param jo
     *            the JSON object.
     * @return the extracted Map.
     * @throws JSONException
     */
    public static Map<String, Set<String>> getMapStringSet(final JSONObject jo)
            throws JSONException
    {
        final Map<String, Set<String>> map = new HashMap<String, Set<String>>();
        final Iterator<String> jkeys = jo.keys();
        while( jkeys.hasNext() )
        {
            final String key = jkeys.next();
            final JSONArray ja = jo.getJSONArray(key);
            final Set<String> components = new HashSet<String>();
            for (int i = 0; i < ja.length(); i++)
            {
                final String icomp = (String) ja.get(i);
                components.add(icomp);
            }
            map.put(key, components);
        }
        return map;
    }

    
    /**
     * Extract a Map with entries (String, String) from a JSON object.
     * 
     * @param jo
     *            the JSON object.
     * @return the extracted Map.
     * @throws JSONException
     */
    public static Map<String, String> getMapStringString(final JSONObject jo)
            throws JSONException
    {
        final Map<String, String> map = new HashMap<String, String>();
        final Iterator<String> itKeys = jo.keys();
        while( itKeys.hasNext() )
        {
            final String key = itKeys.next();
            final String value = jo.getString(key);
            map.put(key, value);
        }
        return map;
    }
    
    /**
     * Check if the input name is to be considered the name of a fits file.
     * @param name name of the file to consider.
     * @return true/false
     */
    public static boolean isConsideredFits(final String name)
    {
        if (name == null)
        {
            logger.warn("NULL filename in input.");
            return false;
        }
        else if (acceptHdrFilesAsFits)
        {
            return (isFitsFile(name) || isFitsHeader(name));
        }
        else
        {
            return isFitsFile(name);
        }
    }

    
    /**
     * Check if the input name is to be considered a fits header file.
     * 
     * @param name
     * @return true if name ends with ".hdr", false otherwise.
     */
    public static boolean isFitsHeader(final String name)
    {
        if (name == null)
        {
            logger.warn("NULL filename in input.");
            return false;
        }
        return name.toLowerCase().endsWith(".hdr");
    }

    
    /**
     * Check if the input is to be considered as a compressed fits file.
     * 
     * @param name
     * @return true if name ends with "fits.fz", false otherwise.
     */
    public static boolean isFitsCompressedFile(final String name)
    {
        if (name == null)
        {
            logger.warn("NULL filename in input.");
            return false;
        }
        return name.toLowerCase().endsWith(".fits.fz");
    }

    
    /**
     * Check if the input name ends with .fits.
     * 
     * @param name
     * @return true if name ends with ".fits" or ".fits.fz", false otherwise.
     */
    public static boolean isFitsFile(final String name)
    {
        if (name == null)
        {
            logger.warn("NULL filename in input.");
            return false;
        }
        return ( isFitsCompressedFile(name) 
                || name.toLowerCase().endsWith(".fits"));
    }
    

    /**
     * Join together in a single string a list of strings.
     * 
     * @param strings
     *            list of strings to join
     * @param separator
     *            Separator to use between two strings in the list when are
     *            joined.
     * @return a single string composed by the input strings separated by the
     *         input separator.
     */
    public static String joinListString(List<String> strings, String separator)
    {
        if (separator == null)
        {
            separator = "";
        }
        if (strings == null)
        {
            strings = Collections.emptyList();
        }
        return StringUtils.join(strings, separator);
    }

    
    /**
     * Read the whole file into a String.
     * 
     * @param f the File to read.
     * @return the string with all the file's content.
     * @throws IOException
     */
    public static String readFile(final File f) throws IOException
    {
        final InputStream is = new FileInputStream(f);
        final BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        String line = null;
        final StringBuilder stringBuilder = new StringBuilder();
        final String ls = System.getProperty("line.separator");
        while( (line = reader.readLine()) != null )
        {
            stringBuilder.append(line);
            stringBuilder.append(ls);
        }
        line = stringBuilder.toString();
        is.close();
        reader.close();
        return line;
    }

    /**
     * Read the whole file into a Set of Strings.
     * 
     * @param f the File to read.
     * @return the string with all the file's content.
     * @throws IOException
     */
    public static Set<String> readAsSet(InputStream is) throws IOException
    {
        final BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        final Set<String> readFileSet = new HashSet<String>();
        String line = null;
        while( (line = reader.readLine()) != null )
        {
            boolean added = readFileSet.add(line);
            if (!added)
            {
                logger.warn("Repeated line - will not be added to the Set: " + line);
            }
        }
        is.close();
        reader.close();
        return readFileSet;
    }
    
    
    
    /**
     * Slice a Set in smaller sets, each not bigger than maxSize.
     * 
     * @param original
     *            the original Set
     * @param maxSize
     *            the max number of elements for each subset. If maxSize is 0 or
     *            less, a list with the original set as single element is
     *            returned.
     * @return List of sub set from the original set.
     */
    public static <T> List<Set<T>> slice(final Set<T> original,
            final int maxSize)
    {
        if (original == null)
        {
            logger.error("Null input argument: original");
            throw new IllegalArgumentException("Null input argument: original");
        }
        final List<Set<T>> slices = new ArrayList<Set<T>>();
        if ((original.size() <= maxSize) || (maxSize <= 0))
        {
            String msg = "Returning a single element list because the input size is "
                    + original.size() + " and max size is ";
            if (maxSize <= 0)
            {
                msg += "non-positive (maxSize=" + maxSize + ")";
            }
            else
            {
                msg += "greater or equal than input size (maxSize=" + maxSize
                        + ")";
            }
            logger.debug(msg);
            slices.add(original);
            return slices;
        }
        final List<T> originalAsList = new ArrayList<T>(original);
        Set<T> partial = null;
        int toIndex = 0;
        int fromIndex = 0;
        while( toIndex < original.size() )
        {
            fromIndex = toIndex;
            toIndex += maxSize;
            if (toIndex > original.size())
            {
                toIndex = original.size();
            }
            partial = new HashSet<T>(originalAsList.subList(fromIndex, toIndex));
            slices.add(partial);
        }
        return slices;
    }

    
    /**Build the list of file path names of the info files. The names are in 
     * {@link Consts#UPDATE_INFO_FILES} and the path is simply the release  
     * base directory.
     * @param reldir the release base directory.
     * @return list of string representing the full path on disk of the files.
     */
    public static List<String> infoFilePaths(String reldir, List<String> infoFileNames)
    {
        List<String> infoFilesList = new ArrayList<String>(); 
        for (final String fileToRemove : infoFileNames)
        {
            String filePath;
            try
            {
                filePath = (new File(reldir + File.separator
                        + fileToRemove)).getCanonicalPath();
                infoFilesList.add(filePath);
            }
            catch( final IOException e )
            {
                logger.error(e.toString());
            }
        }
        return infoFilesList;
    }
    
    
    /**Check if the input File is to be validated. In principle all the files 
     * undergo a validation (even if it's an empty validation). The exceptions 
     * are File objects that are actually sub-directories, File objects which 
     * disappeared from disk (a non-existing file cannot be validated, this will 
     * generate a missing file error later) and the info files describing an 
     * update release. 
     * @param f the file to check. 
     * @param exclusionPathNameList the file that are expected to be on disk but 
     * do not need validation. These are the info files in case of an update release. 
     * @return if the input file is to be validated.
     */
    public static boolean isFileToValidate(File f, List<String>exclusionPathNameList)
    {
        if (f.isDirectory())
        {
            logger.debug("Skipping directory: " + f.getAbsolutePath());
            return false;
        }
        if (!f.exists())
        {
            logger.debug("Skipping non existing file: " + f.getAbsolutePath());
            return false;
        }
        if (exclusionPathNameList.contains(f.getAbsolutePath()))
        {
            logger.debug("Skipping info file: " + f.getAbsolutePath());
            return false;
        }
        return true;
    }
    
    
    /**Return a vector containing all the file pathnames in the directory tree 
     * routed at input dir. Note that only regular files are returned, not 
     * sub-directories.
     * @param dir the base directory.
     * @return Vector with  the found paths.
     * @throws IOException if the base directory or any sub directory cannot be 
     * accessed.
     */
    public static Vector<String> allFilePaths(String dir) throws IOException
    {
        if (dir == null)
        {
            logger.error("Null input argument: dir");
            throw new IllegalArgumentException("Null input argument: dir");
        }

        logger.debug("Retrieving all files on disk in " + dir);
        final RecursiveFileSelectionModel r = new RecursiveFileSelectionModel(dir);
        Vector<String> names = null;
        try
        {
            names = r.getFileList();
        }
        catch (final NullPointerException e)
        {
            throw new IOException(
                    "Unreadable subdirectory. Cannot get list of entries in directory: "
                    + dir);
        }
        return names;
    }
    
    
    /**Check for circular definition in a map of product.
     * @param provenance the map of provenance definition to check. It must have 
     * entries in the format (product, Set of components).
     * @return the Set of provenance keys (i.e. products) which show a circular 
     * dependency. It is an empty Set if there is no circular definition. 
     */
    public static Set<String> circularDefinitions(Map<String, Set<String>> provenance)
    {
        if (provenance == null)
        {
            logger.error("Null input argument: provenance");
            throw new IllegalArgumentException(
                    "Null input argument: provenance");
        }
        DependencyCheck check = new DependencyCheck();
        Set<String> loops = check.findLoops(provenance);
        return loops;
    }
    
    
    /**Encapsulate here the creation of a DataTransportFormatHandler, so that in 
     * case of failure memory can be freed and creation re-attempted.
     * @param fileFullPathName
     * @return the allocated DataTransportFormatHandler object.
     * @throws FileHandlerException when the second attempt fails.
     */
    public static DataTransportFormatHandler allocateDTFH(String fileFullPathName) throws FileHandlerException
    {
    	if (! isConsideredFits(fileFullPathName)) 
            throw new FileHandlerException("Cannot create a fits handler for this non fits/hdr file.");
 
       	DataTransportFormatHandler fh = null;
        try {
            if (ValidationUtil.isFitsHeader(fileFullPathName))
                fh = new HDRHandler(fileFullPathName);
            else {
            	boolean compressed = ValidationUtil.isFitsCompressedFile(fileFullPathName);
               	fh = new FITSHandler(fileFullPathName, compressed);
            }
        } catch (FileHandlerException e) {
            logger.info("Re-trying after exception in creating a fits handler for file : "
                    + fileFullPathName);
            Runtime rt = Runtime.getRuntime();
            long before = rt.freeMemory();
            // It would seem sensible to call runFinalization() before gc() but
            // for unknown reasons it works better with this order: with this 
            // order all the handlers are created at the latest on the second 
            // attempt while using runFinalization() before gc() let few handler 
            // creations fail. (Tested with 40000 hdr files + 40000 fits files).
            rt.gc();
            rt.runFinalization();
            long after = rt.freeMemory();
            logger.info("Call to Java GC was forced. Before free bytes: " 
                    + before +", after: "+ after);
            logger.info("OK, after the GC call the fits handler was successfully created"); 
            if (ValidationUtil.isFitsHeader(fileFullPathName))
            {
                fh = new HDRHandler(fileFullPathName);
            }
            else 
            {
                fh = new FITSHandler(fileFullPathName);
            }
        }
        logger.debug ("DataTransportFormatHandler successfully created for " + fileFullPathName);
        
        return fh;
    }
}
