package org.eso.phase3.validator;

import java.io.FileReader;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.PosixParser;
import org.apache.log4j.Logger;

/**
 * Configure a run of the validation tool.
 * 
 * @author dsforna
 * 
 */
public class ValidatorConfigurationCli implements ValidatorConfiguration
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidatorConfigurationCli.class);

    /**
     * Map for configuration values with entries: (longCli, name). 
     * Example: ("modification-type", "release.modification.type")
     */
    private Map<String, String> cli2conf = null;
    
    /**
     * Map for configuration values with entries: (name, longCli), when longCli exists. 
     * Example: ("release.modification.type", "modification-type"). Just for 
     * error messages.
     */
    private Map<String, String> conf2cli = null;
    
    /** The command line for this run. */
    private CommandLine cml = null;

    /**
     * The configuration is stored in this Properties. The keys are the name
     * fields of the {@link ValidatorConfiguration#OptionNames} defined in
     * {@link ValidatorConfiguration}
     */
    private Properties configuration = null;

    /** Unless all these values are set the validator will refuse to run. */
    private final List<String> mandatoryConfigurationValues;

    /**
     * Map for configuration values with entries: (Option, OptionName). Option
     * object is created from OptionName, but OptionName contains the string
     * used as key in the configuration property. Example: Option contains
     * "modification-type", "m" OptionName contains "modification-type", "m" and
     * "release.modification.type"
     */
    private Map<Option, OptionNames> option2optionName = null;

    /** The recognized options on command line. */
    private Options options = null;

    /**
     * Build the configuration from the input arguments. Afterwards the
     * configuration is checked with a call to {@link#checkConfiguration}.
     * 
     * @throws ValidatorException
     *             if the configuration check fails.
     */
    public ValidatorConfigurationCli(final String[] args)
            throws ValidatorException
    {
        mandatoryConfigurationValues = Arrays.asList(
                // Not configurable from command line (unless using system
                // properties):
                INTERNAL_RUN_CONF_NAME,
                // Configurable from command line:
                RELEASE_DIR.name, FITSVERIFY.name, THREADS.name, URL.name, 
                MODIFICATION_TYPE.name);

        doInternalConfiguration();
        
        final CommandLineParser parser = new PosixParser();
        try
        {
            cml = parser.parse(options, args);
        }
        catch( final Exception e )
        {
            logger.error("Error parsing command line: " + e.toString());
            throw new ValidatorException(e);
        }

        if (cml.getArgs().length > 0)
        {
            final String msg = "Unrecognized option in command line: "
                    + Arrays.toString(cml.getArgs());
            logger.error(msg);
            throw new ValidatorException(msg);
        }

        final Option[] processedOptions = cml.getOptions();

        boolean confFileFromCli = false;
        for (final Option po : processedOptions)
        {
            final String key = po.getLongOpt();
            String value = po.getValue();
            if (value == null)
            {
                value = "";
            }
            if (CONF_FILE_PATH.longCli.equals(key))
            {
                confFileFromCli = true;
            }
            configuration.setProperty(cli2conf.get(key), value);
        }

        if (confFileFromCli)
        {
            final String confFile = configuration.getProperty(CONF_FILE_PATH.name);
            final Properties confFileProperties = new Properties();
            try
            {
                confFileProperties.load(new FileReader(confFile));
            }
            catch( final Exception e )
            {
                logger.error("Error loading configuration file ("
                        + CONF_FILE_PATH.name + " = " + confFile + "). "
                        + e.toString());
                throw new ValidatorException(e);
            }
            logger.debug("Adding to configuration the content of file "
                    + confFile);
            addPropertiesToConfiguration(confFileProperties);
        }

        for (final String confVal : mandatoryConfigurationValues)
        {
            logger.info("Configured value " + confVal + "="
                    + getOptionValue(confVal));
        }
        
        checkConfiguration();
    }

    /**
     * Check if the current configuration is valid.
     * 
     * @throws ValidatorException
     */
    public void checkConfiguration() throws ValidatorException
    {

        if (configuration.containsKey(SHOW.name))
        {
            printConf();
            System.exit(0);
        }
        if (configuration.containsKey(HELP.name))
        {
            printHelp();
            System.exit(0);
        }
        if (configuration.containsKey(VERSION.name))
        {
            printMetaConf();
            System.exit(0);
        }

        if (configuration.containsKey(VERBOSE_MODE.name))
        {
            ValidationUtil.setDebugLog(true);
        }
        
        
        boolean missingValue = false;
        String missingNeededMsg = "";
        for (final String key : mandatoryConfigurationValues)
        {
            if (configuration.getProperty(key) == null)
            {
                String errMsg = "Missing configuration value";
                if (conf2cli.containsKey(key))
                {
                    errMsg+= " from command line: " + conf2cli.get(key); 
                }
                errMsg+= " ("+key+"). "; 
                    
                logger.error(errMsg);
                missingNeededMsg += errMsg;
                missingValue = true;
            }
        }
        if (missingValue)
        {
            throw new ValidatorException(missingNeededMsg);
        }

        final String modificationType = configuration.getProperty(
                ValidatorConfiguration.MODIFICATION_TYPE.name).trim();

        if (modificationType.toLowerCase().equals(
                ValidatorConfiguration.MODIFICATION_TYPE_VALUE_UPDATE.toLowerCase()))
        {
            logger.debug("modification type input " + modificationType
                    + " is the allowed possible value: "
                    + ValidatorConfiguration.MODIFICATION_TYPE_VALUE_UPDATE);
            configuration.setProperty(
                    ValidatorConfiguration.MODIFICATION_TYPE.name,
                    ValidatorConfiguration.MODIFICATION_TYPE_VALUE_UPDATE);
        }
        else if (modificationType.toLowerCase().equals(
                ValidatorConfiguration.MODIFICATION_TYPE_VALUE_NEW.toLowerCase()))
        {
            logger.debug("modification type input " + modificationType
                    + " matches with possible value:"
                    + ValidatorConfiguration.MODIFICATION_TYPE_VALUE_NEW);
            configuration.setProperty(
                    ValidatorConfiguration.MODIFICATION_TYPE.name,
                    ValidatorConfiguration.MODIFICATION_TYPE_VALUE_NEW);
        }
        else
        {
            final String msg = 
                    "Cannot understand the modification type for this release ("
                    + modificationType
                    + "). Please give as input either "
                    + ValidatorConfiguration.MODIFICATION_TYPE_VALUE_NEW
                    + " or "
                    + ValidatorConfiguration.MODIFICATION_TYPE_VALUE_UPDATE;
            logger.error(msg);
            throw new ValidatorException(msg);
        }
        
        final String numberOfThread = configuration.getProperty(
                ValidatorConfiguration.THREADS.name).trim();
        try 
        {
            Integer t = Integer.parseInt(numberOfThread);
            if (t <0) 
            {
                String msg = "Negative number of threads the input (" 
                        + numberOfThread 
                        + "). Please specify as number greate or equal to 0.";
                logger.error(msg);
                throw new ValidatorException(msg);
            }
        }
        catch (NumberFormatException e)
        {
            String msg = "Error interpreting as number of threads the input (" 
                    + numberOfThread +") : " ;
            logger.error(msg + e.toString());
            throw new ValidatorException(msg + e.getMessage());
        }
        logger.debug("Full Configuration: " 
                + Arrays.toString(configuration.entrySet().toArray()) );
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.validator.ValidatorConfiguration#getString(java.lang.String
     * )
     */
    @Override
    public String getOptionValue(final String option)
    {
        if (option == null)
        {
            logger.error("Null input argument: option");
            throw new IllegalArgumentException("Null input argument: option");
        }
        String val = configuration.getProperty(option);
        if (val == null)
        {
            logger.debug("Value for " + option
                    + " not set in configuration, returning the empty string.");
            val = "";
        }
        return val;
    }

    /**
     * If the option's value is not defined (null) it is considered false,
     * otherwise, any string starting with the character "t" or "T" (
     * {@link ValidatorConfiguration#OPTION_TRUE}) is considered true and any
     * other string is considered false.
     * 
     * @see org.eso.phase3.validator.ValidatorConfiguration#isTrue(java.lang.String)
     */
    public boolean isTrue(final String option)
    {
        if (option == null)
        {
            logger.error("Null input argument: option");
            throw new IllegalArgumentException("Null input argument: option");
        }
        final String value = configuration.getProperty(option);
        if (value == null)
        {
            return false;
        }
        return value.toUpperCase().startsWith(OPTION_TRUE);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.validator.ValidatorConfiguration#setOptionValue(java.lang
     * .String)
     */
    @Override
    public void setOptionValue(final String optionName, final String optionValue)
            throws ValidatorException
    {
        if (optionName == null)
        {
            logger.error("Null input argument: optionName");
            throw new IllegalArgumentException(
                    "Null input argument: optionName");
        }

        logger.debug("Programmmatically setting configuration option "
                + optionName + "=" + optionValue);
        configuration.setProperty(optionName, optionValue);
        checkConfiguration();
    }

    /**
     * Add (or replace) to this configuration the names in input if they have a
     * values in the input Properties and are allowed to be set from command
     * line. Note that the input properties can have entries with both the
     * configuration option names or command line names, this method provides
     * the conversion and stores in the configuration the option name (and not
     * the command line name).
     * 
     * @param source
     *            the input Properties.
     */
    private void addPropertiesToConfiguration(final Properties source)

    {
        final List<OptionNames> opNames = new ArrayList<OptionNames>();
        for (final Option opt : (Collection<Option>) options.getOptions())
        {
            opNames.add(option2optionName.get(opt));
        }

        for (final OptionNames opn : opNames)
        {
            if (opn == null)
            {
                final String msg = "NULL option object in input.";
                logger.error(msg);
                throw new IllegalArgumentException(msg);
            }

            if (opn.longCli == null)
            {
                final String msg = "NULL long command line name for "
                        + opn.name;
                logger.error(msg);
                throw new IllegalArgumentException(msg);
            }

            String valueConf = null;
            String keyConf = null;
            if ((opn.longCli != null) && (source.containsKey(opn.longCli)))
            {
                valueConf = source.getProperty(opn.longCli);
                keyConf = opn.name;
            }
            else if ((opn.shortCli != null)
                    && (source.containsKey(opn.shortCli)))
            {
                valueConf = source.getProperty(opn.shortCli);
                keyConf = opn.name;
            }
            else if (source.containsKey(opn.name))
            {
                valueConf = source.getProperty(opn.name);
                keyConf = opn.name;
            }
            else
            {
                logger.trace("value not present in config file for " + opn.name);
                continue;
            }

            if (valueConf == null)
            {
                valueConf = "";
            }
            logger.debug("Setting (from conf file) configuration value "
                    + keyConf + "=" + valueConf);
            configuration.setProperty(keyConf, valueConf);
        }
    }

    /**
     * Add to this configuration the relevant values set from System Properties.
     * Values from system properties are allowed as "back door" configuration, 
     * a user should use command line or file options instead.
     * 
     * @throws ValidatorException
     */
    private void addSystemPropertiesToConfiguration() throws ValidatorException
    {
        List<String> stringValueFromSys = Arrays.asList(
                MOCKUP_FILE_CONF_NAME);

        List<String> booleanFromSys = Arrays.asList(
                INTERNAL_RUN_CONF_NAME, 
                ACCEPT_LEADING_0_IN_KEYWORD_INDEXES_NAME, 
                ENABLE_HDR_FILES);
        
        String spconf = null;
        
        for (String sysOption: booleanFromSys)
        {
            spconf = System.getProperty(sysOption);
            if (spconf != null)
            {
                if (spconf.toUpperCase().startsWith(OPTION_TRUE))
                {
                    configuration.setProperty(sysOption, OPTION_TRUE);
                }
                else if (spconf.toUpperCase().startsWith(OPTION_FALSE))
                {
                    configuration.setProperty(sysOption, OPTION_FALSE);
                }
                else
                {
                    String msg = "Unrecognized sysyem property value for "
                            + sysOption
                            + " (Set one of T,t,True,true,F,f,False,false or leave it unset).";
                    logger.error(msg);
                    throw new ValidatorException(msg);
                }
            }
        }

        for (String sysOption: stringValueFromSys)
        {
            spconf = System.getProperty(sysOption);

            if (spconf != null)
            {
                configuration.setProperty(sysOption, spconf);
            }
        }
    }

    /**
     * Create a command line option from the input OptionNames. At the same time
     * stores in {@link#cli2conf} the mapping between the command line name and
     * the OptionNames name.
     * 
     * @param confOption
     *            the input OptionNames.
     * @return the created command line option.
     */
    private Option createOption(final OptionNames confOption)
    {
        final Option o = new Option(confOption.shortCli, confOption.longCli,
                confOption.hasArg, confOption.description);
        option2optionName.put(o, confOption);
        cli2conf.put(confOption.longCli, confOption.name);
        conf2cli.put(confOption.name, confOption.longCli);
        return o;
    }

    /**
     * Configure if ".hdr" files are to be accepted as fits or not.
     * If the configuration value is not set the default {@link Consts#DEFAULT_ENABLE_HDR_FILES}
     * is used.
     */
    private void configureHdrFiles()
    {
        boolean val = Consts.DEFAULT_ENABLE_HDR_FILES;
        
        if (! getOptionValue(ValidatorConfiguration.ENABLE_HDR_FILES).equals(""))
        {
            val = isTrue(ValidatorConfiguration.ENABLE_HDR_FILES);
        }
        ValidationUtil.setAcceptHdrFilesAsFits(val);
    }
    
    /**
     * Load the internal configuration file (it defines the default
     * configuration values) and the relevant System Properties if they are set.
     * 
     * @throws ValidatorException
     */
    private void doInternalConfiguration() throws ValidatorException
    {
        cli2conf = new HashMap<String, String>();
        conf2cli= new HashMap<String, String>();
        option2optionName = new HashMap<Option, OptionNames>();
        configuration = new Properties();

        try
        {
            final InputStream is = getClass().getResourceAsStream(
                    INTERNAL_CONF_FILE);
            configuration.load(is);
            is.close();
            String confStr = "";
            final Enumeration keys = configuration.propertyNames();
            while( keys.hasMoreElements() )
            {
                final String key = (String) keys.nextElement();
                final String value = configuration.getProperty(key);
                confStr += key + "=" + value + " ";
            }
            logger.debug("Loaded from internal configuration file: " + confStr);
        }
        catch( final Exception e )
        {
            logger.error("Cannot load as resource the internal configuration file ("
                    + INTERNAL_CONF_FILE + ")");
            throw new ValidatorException(e);
        }

        addSystemPropertiesToConfiguration();

        options = new Options();
        options.addOption(createOption(RELEASE_DIR));
        options.addOption(createOption(FITSVERIFY));
        options.addOption(createOption(THREADS));
        options.addOption(createOption(MODIFICATION_TYPE));
        options.addOption(createOption(VERSION));
        options.addOption(createOption(HELP));
        options.addOption(createOption(SHOW));
        options.addOption(createOption(VERBOSE_MODE));
        options.addOption(createOption(CONF_FILE_PATH)); // TODO: should be allowed only for internal run?
        if (isTrue(INTERNAL_RUN_CONF_NAME))
        {
            options.addOption(createOption(URL));
        }
        
        configureHdrFiles();
    }

    private void printConf()
    {
        logger.info("Display current configuration.");
        final Enumeration keys = configuration.propertyNames();
        while( keys.hasMoreElements() )
        {
            final String key = (String) keys.nextElement();
            System.out.println(key + " = " + configuration.getProperty(key));
        }
    }

    private void printHelp()
    {
        final HelpFormatter formatter = new HelpFormatter();
        formatter.setWidth(80);
        formatter.printHelp(getOptionValue(APPLICATION_COMMAND), options);
    }

    private void printMetaConf()
    {
        for (final String key : Arrays.asList(APPLICATION_NAME,
                APPLICATION_COMMAND, APPLICATION_VERSION))
        {
            System.out.println(key + " = " + configuration.getProperty(key));
        }
    }

}
