package org.eso.phase3.validator;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.eso.oca.fits.DataTransportFormatHandler;

/**
 * This class encapsulates the creation of the validator's sub-objects which
 * might need to be replaced with mockup objects during testing. Real objects
 * and mockup objects need to implement a common interface and are seen by the
 * external clients only through this interface. The type name of the interface
 * and of the real object is hard-coded, while the type name of the
 * corresponding mockup object is configured in the mockup properties file (this
 * allows for different mockup objects in different tests). Each object has its
 * own create() method which instantiates either of the real/mockup and returns
 * the common interface.
 * 
 * @author dsforna
 */
public class ValidatorFactory
{
    public static final String HTTP_CONF = "org.eso.phase3.validator.HttpConfImp";

    /** Following Classes  can be replaced by a mockup. */
    public static final String RELEASE_PARSER = "org.eso.phase3.validator.ReleaseParserImp";
    public static final String VALIDATOR_MANAGER = "org.eso.phase3.validator.ValidatorManagerImp";
    public static final String VALIDATOR_OCA_PARSER = "org.eso.phase3.validator.ValidatorOcaParserImp";

    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidatorFactory.class);

    private Properties mockups = null;

    private ValidatorSetup setup = null;

    /**
     * Constructor. Set up the mockup mapping (if any) from the configuration.
     * 
     * @param config
     *            the configuration.
     */
    public ValidatorFactory(final ValidatorSetup setup)
    {
        if (setup == null)
        {
            logger.error("Null input argument: setup");
            throw new IllegalArgumentException("Null input argument: setup");
        }

        this.setup = setup;
        final ValidatorConfiguration config = setup.getConfiguration();
        if (config == null)
        {
            logger.error("Null configuration object.");
            throw new IllegalArgumentException("Null configuration object.");
        }
        
        if (config.isTrue(ValidatorConfiguration.ACCEPT_LEADING_0_IN_KEYWORD_INDEXES_NAME))
        {
            FitsReleaseStructureParserImp.acceptLeading0InIndexes(true);
        }
        

        final String mockupPropertiesFile = config.getOptionValue(
                ValidatorConfiguration.MOCKUP_FILE_CONF_NAME);
        if ((mockupPropertiesFile == null) || (mockupPropertiesFile.equals("")))
        {
            return;
        }
        
        try
        {
            InputStream is = ClassLoader.getSystemResourceAsStream(mockupPropertiesFile);
            if (is == null)
            {
                logger.debug(
                        "Cannot find as resource the mockup properties. Trying as disk file ["
                        + mockupPropertiesFile + "]");
                is = new FileInputStream(new File(mockupPropertiesFile));
            }

            if (is == null)
            {
                logger.warn(
                        "The mockup properties could not be found as resource nor as file ["
                        + mockupPropertiesFile + "]");
                return;
            }

            logger.info("Reading the mockup properties [" + mockupPropertiesFile + "]");
            mockups = new Properties();
            mockups.load(is);
            is.close();
        }
        catch( final Exception e )
        {
            logger.error(e.toString());
            mockups = null;
        }
    }

    /**
     * Method encapsulating the creation of the used {@link#ArchivedFilesClient}
     * for this run. Note that the ArchivedFilesClient started with the idea of 
     * being a component connecting with a remote server  (ArchivedFilesServer)
     * but currently is not used and therefore an object of the NullArchivedFilesClient
     * class is instantiated. Maybe in the future this object could actually connect
     * to the database when the validator runs locally at ESO.
     * 
     * @return a NullArchivedFilesClient object.
     */
    public ArchivedFilesClient createArchivedFilesClient()
    {
        final ValidatorConfiguration config = setup.getConfiguration();
        return new NullArchivedFilesClient(config);
    }

    /**
     * Method encapsulating the creation of the used {@link#HttpConf} for this
     * run.
     * 
     * @return Either the real HttpConf or the mockup HttpConf, according to
     *         what is specified in this configuration object.
     */
    public HttpConfMHI createHttpConf() throws ValidatorException
    {
        try
        {
            ///return new HttpConfMH(setup.getConfiguration());
        	String ocaBaseUrl = setup.getConfiguration().getOptionValue(ValidatorConfiguration.URL.name);
        	return new HttpConfMultipleHeadersImpl(ocaBaseUrl);
        }
        catch( Exception e )
        {
            throw new ValidatorException(e);
        }
    }

    /**
     * Method encapsulating the creation of the used {@link#ReleaseParser} for
     * this run. A ReleaseParser is instantiated in case of validation of a new
     * (modification type CREATE) release.
     * 
     * @return Either the real ReleaseParser or the mockup ReleaseParser,
     *         according to what is specified in the input configuration object.
     * @throws ValidatorException
     * @throws IOException
     */
    public ReleaseParserWithExtraInfo createParser() throws IOException, ValidatorException
    {
        final ValidatorConfiguration config = setup.getConfiguration();
        final String constructorParameter = config.getOptionValue(ValidatorConfiguration.RELEASE_DIR.name);
        final String mockupClassName = getMockup(RELEASE_PARSER);
        if (mockupClassName == null)
        {
        	String fv = config.getOptionValue(ValidatorConfiguration.FITSVERIFY.name);
            ReleaseParserImp rp = new ReleaseParserImp(constructorParameter);
            if (fv.equals(ValidatorConfiguration.FITSVERIFY.defaultValue)) {
            	rp.setCheckErrorFiles(true);
            }
            return rp;
        }
        logger.info("Creating a mockup parser of type: " + mockupClassName);
        try
        {
            final Class<?> mockupClass = Class.forName(mockupClassName);
            final Constructor<?> constructor = mockupClass.getConstructor(String.class);
            return (ReleaseParserWithExtraInfo) constructor.newInstance(constructorParameter);
        }
        catch( final Exception e )
        {
            logger.error(e.toString());
            throw new ValidatorException(e);
        }
    }

    
    /**
     * Method encapsulating the creation of the used {@link#ReleaseUpdateParser}
     * for this run. A ReleaseUpdateParser is instantiated in case of validation
     * of an UPDATE release.
     * 
     * @return the update parser
     * @throws ValidatorException
     * @throws IOException
     */
    public ReleaseParserWithExtraInfo createUpdateParser()
            throws ValidatorException, IOException
    {
        final ValidatorConfiguration config = setup.getConfiguration();
        final String constructorParameter = config.getOptionValue(ValidatorConfiguration.RELEASE_DIR.name);
        return new ReleaseUpdateParser(constructorParameter);
    }

    
    /**
     * Method encapsulating the creation of the used {@link#ValidatorManager}
     * for this run.
     * 
     * @return Either the real ValidatorManager or the mockup ValidatorManager,
     *         according to what is specified in this configuration object.
     */
    public ValidatorManager createValidatorManager() throws ValidatorException
    {
        final String mockupClassName = getMockup(VALIDATOR_MANAGER);
        if (mockupClassName == null)
        {
            return new ValidatorManagerImp(setup);
        }
        logger.info("Creating a mockup object of type: " + mockupClassName);
        try
        {
            final Class<?> mockupClass = Class.forName(mockupClassName);
            final Constructor<?> constructor = mockupClass.getConstructor(ValidatorSetup.class);
            return (ValidatorManager) constructor.newInstance(setup);
        }
        catch( final Exception e )
        {
            logger.error(e.toString());
            throw new ValidatorException(e);
        }
    }

    /**
     * Method encapsulating the creation of the used {@link#ValidatorOcaParser}
     * for this run.
     */
    public ValidatorOcaParser createValidatorOcaParser(
            final String fileFullPathName, final String extCategory, DataTransportFormatHandler fh, 
            final int headerIndex, final int offset) throws ValidatorException
    {
        
        final HttpConfMHI httpConf = setup.getHttpConf();
        return new ValidatorMHOcaCachedParser(httpConf,
                fileFullPathName, extCategory, fh, headerIndex, offset);
    }

    /**
     * @param realClassName
     *            the real class name.
     * @return if a different class is defined as mockup return its name,
     *         otherwise return null.
     */
    private String getMockup(final String realClassName)
    {
        if (mockups == null)
        {
            return null;
        }
        final String mockupClassName = mockups.getProperty(realClassName);
        if (mockupClassName == null)
        {
            return null;
        }
        if (realClassName.equals(mockupClassName))
        {
            return null;
        }
        return mockupClassName;
    }
}
