package org.eso.phase3.validator;

import java.io.File;
import org.apache.log4j.Logger;

/**
 * A file known by the validator application either because defined in the
 * release structure or because found on disk. Only files which are both in the
 * release structure and on disk (or on disk remotely) can be valid.
 * 
 * @author dsforna
 */
public class ValidatorFile
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidatorFile.class);

    /**
     * Static factory to create a ValidatorFile object for a file present in the 
     * original release structure. Used in case of an UPDATE release.
     * @param name name of the file for which this object is created.
     * @param category category of the file for which this object is created.
     * @return the created ValidatorFile object.
     */
    public static ValidatorFile fileInOriginalReleaseStructure(
            final String name, final String category)
    {
        final ValidatorFile vf = new ValidatorFile();
        vf.category = category;
        vf.isInReleaseStructure = true;
        vf.name = name;
        vf.validationStep = NullValidator.validStep;
        return vf;
    }

    /**
     * Static factory to create a new object corresponding to a file in the 
     * release structure but not found on disk.
     * @param name
     *            the file name
     * @param category
     *            the category of the file
     * @return the created ValidatorFile object.
     */
    public static ValidatorFile fileOnlyInReleaseStructure(final String name,
            final String category)
    {
        final ValidatorFile vf = new ValidatorFile();
        vf.category = category;
        vf.isInReleaseStructure = true;
        vf.name = name;
        vf.validationStep = NullValidator.invalidStep;
        vf.errorMsg = vf.name
                + ": File in the release structure not found on disk";
        logger.debug("Returning new invalid fileOnlyInReleaseStructure for "
                + vf.name);
        return vf;
    }

    /**
     * Static factory to create a new object corresponding to a file found on 
     * disk but not present in the release structure.
     * @param f
     *            the file object
     * @return the created object.
     */
    public static ValidatorFile fileOnlyOnDisk(final File f)
    {
        final ValidatorFile vf = new ValidatorFile();
        vf.name = f.getName();
        vf.validationStep = NullValidator.invalidStep;
        vf.isInReleaseStructure = false;
        vf.errorMsg = vf.name
                + ": File on disk not found in the release structure";
        logger.debug("Returning new invalid fileOnlyOnDisk for " + vf.name);
        return vf;
    }

    /**
     * Create a new object corresponding to a file found both on disk and in the
     * release structure.
     * 
     * @param category
     *            the category of the file
     * @param f
     *            the file object
     * @return the created object.
     */
    public static ValidatorFile fileReleaseDisk(final String category,
            final File f)
    {
        final ValidatorFile vf = new ValidatorFile();
        vf.name = f.getName();
        vf.category = category;
        vf.isInReleaseStructure = true;
        logger.debug("Returning new fileReleaseDisk for " + vf.name);
        return vf;
    }

    private String category = null;
    private String errorMsg = null;
    private boolean isInReleaseStructure = false;
    private String name = null;
    private ValidationStep validationStep = null;

    /**
     * Constructor is private so new objects are created only through the 
     * provided static factories.
     */
    private ValidatorFile()
    {
    }


    /**
     * @return the category of the file associated to this object.
     */
    public String getCategory()
    {
        return category;
    }

    /**
     * @return the error message of this object.
     */
    public String getError()
    {
        if (errorMsg != null)
        {
            return errorMsg;
        }
        return validationStep.validationReport().getError();
    }

    /**
     * @return the name of the file associated to this object.
     */
    public String getName()
    {
        return name;
    }

    /**
     * @return the validationStep which is to be run on the file associated to this object.
     */
    public ValidationStep getValidationStep()
    {
        return validationStep;
    }

    /**
     * @return if the file of this object is in the release structure.
     */
    public boolean isInReleaseStructure()
    {
        return isInReleaseStructure;
    }

    /**
     * @param validationStep
     *            the validationStep to set
     */
    public void setValidationStep(final ValidationStep validationStep)
    {
        this.validationStep = validationStep;
    }

}
