package org.eso.phase3.validator;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.log4j.Logger;
import org.eso.oca.fits.DataTransportFormatHandler;
import org.eso.oca.fits.DefaultFileFilter;
import org.eso.oca.fits.KeywordNotFoundException;
import org.eso.oca.fits.OCAFile;
import org.eso.oca.fits.OCAFileFilter;
import org.eso.oca.fits.UndefinedValueException;
import org.eso.oca.parser.ASTStart;
import org.eso.oca.parser.InterpretationException;
import org.eso.oca.parser.OcaParser;
import org.eso.oca.parser.ParseException;
import org.eso.phase3.validator.HttpConfMHI.OcaRulesKey;

/**
 * Implement the OCA parser for multiple headers of the same fits file, with caching, 
 * so that an OCA rule file is read only once to create the parser. The key to 
 * identify a OCA parser is no longer the category but the combination 
 * (category, offset) where offset is the difference between the index of the 
 * header to parse (passed at construction together with the fits file from where 
 * to parse the header) and the index of the primary header of the fits file.   
 * 
 * @author dsforna
 * 
 */
public class ValidatorMHOcaCachedParser implements ValidatorOcaParser
{
    
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidatorMHOcaCachedParser.class);

    /**
     * The cached keyword for this (OcaRulesKey,offset).
     */
    private static final ConcurrentMap<String, String[]> keywords = 
            new ConcurrentHashMap<String, String[]>();
            
    /**
     * The cached parser for this (OcaRulesFilename,offset).
     */
    private static final ConcurrentMap<String, ASTStart> parserStarts = 
            new ConcurrentHashMap<String, ASTStart>();

    /**Category of file <code>fileFullPathName</code>*/
    private final String category;

    /**
     * Store the message in case of error creating or using the OCA parser for 
     * the input file <code>fileFullPathName</code>
     */
    private String errorMsg;

    /**Index of the header to parse (index is in range 0..HDU-1).*/
    private final int headerIndex;

    /**Full pathname on disk of the FITS file to parse.*/
    private final String fileFullPathName;
    private final DataTransportFormatHandler dtfh;

    /**Applications' object to retrieve the OCA rules for this category and 
     * offset index.*/
    private final HttpConfMHI httpConf;

    /**
     * OCA file factory, to generate the OCA file from <code>fileFullPathName</code>
     */
    private OCASingleFileFactory osff;

    /**If this parser could be successfully created.*/
    private boolean valid;

    /**Offset between this headerIndex and the index of the primary header. 
     * If  this headerIndex is the primary header index, the offset is 0.*/
    private final OcaRulesKey ocaRulesKey;

    /**Create an OCA parser.
     * @param config
     * @param httpConf
     * @param fileFullPathName FITS file.
     * @param category category of the FITS file.
     * @param headerIndex index of header of the input FITS to be parsed.
     * @param offsetFromPrimaryHeaderIndex offset of headerIndex from the primary 
     * header index (i.e. the logical extension number, opposed to the physical 
     * extension number).
     */
    public ValidatorMHOcaCachedParser(
            final HttpConfMHI httpConf, 
            final String fileFullPathName,
            final String category, 
            final DataTransportFormatHandler fh, 
            final int headerIndex,
            final int offsetFromPrimaryHeaderIndex)
    {
        ocaRulesKey = new OcaRulesKey(
                new String[]{Consts.CATG_KW}, new String[]{category}, 
                offsetFromPrimaryHeaderIndex);
        this.fileFullPathName = fileFullPathName;
        this.category = category;
        this.dtfh = fh;
        this.headerIndex = headerIndex;
        this.httpConf = httpConf;
        errorMsg = "";
        parseRules();
    }

    
    private String parseRulesFileName() throws ValidatorException, IOException, ParseException, InterpretationException, KeywordNotFoundException, UndefinedValueException
    {
     
        final String[] requestedData = new String[0];
        final String[] dirAndFile = OCASingleFileFactory.splitFullPath(fileFullPathName);
        final String fileName = dirAndFile[1];
        OCAFile[] ocaFiles = null;
        final OCAFileFilter allFilesFilter = new DefaultFileFilter();
        final ArrayList<OCAFile> readFiles = new ArrayList<OCAFile>();
        final ArrayList<File> skippedFiles = new ArrayList<File>();

        Map<String, String> extraKeyword = new HashMap<String, String>();
        extraKeyword.put(Consts.CATG_KW, category);
        extraKeyword.put(Consts.EXTNUM, String.valueOf(ocaRulesKey.offset));
        
        osff = new OCASingleFileFactory(dirAndFile[0], fileName, dtfh);
        osff.createOCAFilesWithExtraKeywords(
                extraKeyword, 
                headerIndex,
                requestedData,allFilesFilter, readFiles, skippedFiles);
        
        ocaFiles = readFiles.toArray(new OCAFile[0]);
        if (ocaFiles.length == 0)
        {
            logger.debug("No OCA rules to be applied for "+ fileName + "(" 
                    + fileFullPathName + ")");
            return null;
        }

        if (ocaFiles.length > 1)
        {
            final String msg = "Too many OCA rules files ("
                    + ocaFiles.length + ")";
            logger.error(msg);
            errorMsg = msg;
            valid = false;
        }

        OCAFile ocaFile = ocaFiles[0];
        if (ocaFile == null)
        {
            final String msg = "Cannot retrieve OCA rules file for "
                    + fileName + "(" + fileFullPathName + ")";
            logger.error(msg);
            errorMsg = msg;
            valid = false;
        }

        if (!valid)
        {
            throw new ValidatorException(errorMsg);
        }

        final ASTStart start = getParserStart();
        ocaFiles = start.classify(new OCAFile[] { ocaFile });
        ocaFile = ocaFiles[0];
        /*******************************************/
        /* VFO: I don't understand this part: the info shold probably be taken directly from the OCAFile */
        String val = "";
        for (final Iterator it = start.getMetaKeywordCorpus().iterator(); it.hasNext();)
        {
            final String kw = (String) it.next();
            if (kw.equals(Consts.OCAFILE))
            {
                val = ocaFile.getKeywordValue(kw);
                break;
            }
        }
        if (val.equals("")) {
            val = null;
            logger.debug(fileFullPathName + " does not need to be validated with OCA rules");
        } else {
            logger.debug(fileFullPathName + " is validated with OCA rules: " + val);
        }
        /****************************************/
        return val;
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorOcaParser#getClassification()
     */
    @Override
    public Map<String, String> getClassification(boolean internal) 
        throws IOException, ParseException, InterpretationException, 
            KeywordNotFoundException, UndefinedValueException, ValidatorException
    {
        logger.trace("");
        if (!valid)
        {
            throw new ValidatorException("Invalid OCA Parser: " + errorMsg);
        }
        final String[] requestedData = keywords.get(ocaRulesKey.getAssociateOcaRuleFile());
        if (requestedData == null)
        {
            errorMsg = "Null requestedData for " + ocaRulesKey.getAssociateOcaRuleFile();
            valid = false;
            logger.error(errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }
       
        final Map<String, String> classification = new HashMap<String, String>();
        final String[] dirAndFile = OCASingleFileFactory.splitFullPath(fileFullPathName);
        // String directoryName = dirAndFile[0];
        final String fileName = dirAndFile[1];
        OCAFile[] ocaFiles = null;
        // The default filter let all files pass.
        final OCAFileFilter allFilesFilter = new DefaultFileFilter();
        final ArrayList<OCAFile> readFiles = new ArrayList<OCAFile>();
        final ArrayList<File> skippedFiles = new ArrayList<File>();
        osff = new OCASingleFileFactory(dirAndFile[0], fileName, dtfh);
        osff.createOCAFiles(headerIndex, requestedData, allFilesFilter, readFiles, skippedFiles);

        ocaFiles = readFiles.toArray(new OCAFile[0]);
        if (ocaFiles.length == 0) {
            final String msg = "Cannot retrieve Metadata structure from file (" + fileFullPathName + ")";
            logger.error(msg);
            errorMsg = msg;
            valid = false;
        }

        if (ocaFiles.length > 1) {
            final String msg = "Too many retrieved Metadata structures (" + ocaFiles.length + ")";
            logger.error(msg);
            errorMsg = msg;
            valid = false;
        }

        OCAFile ocaFile = ocaFiles[0];
        if (ocaFile == null) {
            final String msg = "Cannot build metadata structure for " + fileName + "(" + fileFullPathName + ")";
            logger.error(msg);
            errorMsg = msg;
            valid = false;
        }
        
        if (internal)
        	ocaFile.setCard("P3ORIG", "IDP");
        else
        	ocaFile.setCard("P3ORIG", "EDP");

        if (!valid) {
            throw new ValidatorException(errorMsg);
        }

        final ASTStart start = getParserStart();
        ocaFiles = start.classify(new OCAFile[] { ocaFile });
        ocaFile = ocaFiles[0];
        logger.debug(ocaFile.getFilename()
                + " has been classified. Extract the metakeywords.");

        String msg = ocaFile.getFilename() + " has classification: | ";
        for (final Iterator it = start.getMetaKeywordCorpus().iterator(); it.hasNext();)
        {
            final String kw = (String) it.next();
            final String val = ocaFile.getKeywordValue(kw);
            msg += kw + "=" + val + " | ";
            classification.put(kw, val);
        }
        logger.debug(msg);
        return classification;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorOcaParser#getErrorMsg()
     */
    @Override
    public String getErrorMsg()
    {
        return errorMsg;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorOcaParser#isValid()
     */
    @Override
    public boolean isValid()
    {
        return valid;
    }

    /**Create or get from cache the ASTStart object corresponding to a OCA rules file.
     * @return ASTStart 
     *          object for this category.
     * @throws IOException 
     *          if an InputStream cannot be created from the category rule file.
     * @throws ParseException the local OcaParser object can throw in while 
     * creating the ASTStart object.   
     */
    private synchronized ASTStart getParserStart() throws IOException, ParseException
    {
        logger.trace("");
        if (!httpConf.isValid(category))
        {
            throw new IOException(category 
                    + " is an invalid category: cannot open its metadata rules.");
        }
        
        if (!parserStarts.containsKey(ocaRulesKey.getAssociateOcaRuleFile()))
        {
            logger.debug("Creating OCA parser for: " 
                    + ocaRulesKey.getAssociateOcaRuleFile());
            
            InputStream is = httpConf.getFileStream(ocaRulesKey.getAssociateOcaRuleFile());
            final OcaParser parser = new OcaParser(is);
            parserStarts.putIfAbsent(ocaRulesKey.getAssociateOcaRuleFile(), parser.Start());
        }
        return parserStarts.get(ocaRulesKey.getAssociateOcaRuleFile());
    }

    
    private void parseRulesFromFile(String filename) throws IOException, ParseException
    {
        
        if ( (filename == null) || (filename.equals("")))
        {
            logger.debug("Empty rule filename - nothing to parse.");
            return;
        }
        
        if (keywords.containsKey(filename))
        {
            logger.debug("Rules for: " + filename + " were already parsed");
            return;
        }
        
        logger.info("Parsing OCA rules for:" + filename);
        final ASTStart start = getParserStart();
        final String[] requestedData = start.getRequestedData();
        keywords.putIfAbsent(filename, requestedData);
        logger.debug("Rules parsed and keywords cached for " + filename);
    }
        
    /**
     * As part of the construction, parse the OCA rules for this OCA rules files.
     */
    private void parseRules()
    {
        
        final String methodName = "ValidatorOcaCachedParser::parseRules";
        logger.trace(methodName);
        String selectRulesFile = httpConf.getSelectionRulesFileName(category);
        try
        {
            valid = true;
            ocaRulesKey.setAssociateOcaRuleFile(selectRulesFile);
            parseRulesFromFile(ocaRulesKey.getAssociateOcaRuleFile());
            String rulesFileName = parseRulesFileName();
            ocaRulesKey.setAssociateOcaRuleFile(rulesFileName);
            parseRulesFromFile(rulesFileName);
        }
        catch( final Exception e )
        {
            // DFS09582 use a more significative message to present to the user:
            errorMsg = "Error parsing metadata rules for file " 
                    +  fileFullPathName + " , header index = "+ headerIndex
                    + " - Details are reported in the log file.";
            logger.error(e.toString());
            ocaRulesKey.setAssociateOcaRuleFile(null);
            valid = false;
        }
    }


    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidatorOcaParser#hasRulesFile()
     */
    @Override
    public boolean hasRulesFile()
    {
        String methodName ="ValidatorOcaParser::hasRulesFile";
        logger.trace(methodName);
        return (ocaRulesKey.getAssociateOcaRuleFile() != null);
    }
    
    @Override
    public String getRulesFilename()
    {
        String methodName ="ValidatorOcaParser::getrulesFilename";
        logger.trace(methodName);
        return ocaRulesKey.getAssociateOcaRuleFile();
    }
}

