package org.eso.phase3.validator;

import java.io.File;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eso.phase3.validator.Util;
import org.eso.phase3.validator.Consts.ValidationPhase;
import org.eso.phase3.validator.ValidationReport.STATUS;
import org.eso.phase3.validator.ValidatorStat.StatType;

/**
 * This class contains the "business rules" for the process of the validation. 
 * The implementation details of the file validation are as much as possible 
 * encapsulated in the implementations of {@link ValidationStep}. 
 *  
 * @author dsforna
 *
 */
public class ValidatorManagerImp implements ValidatorManager
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidatorManagerImp.class);

    /** Elements in this list are files already archived at ESO. */
    private Set<String> archived;

    /** This is a sub object of setup, made explicit just for convenience. */
    private final ReleaseParserWithExtraInfo releaseParser;

    /**Main validation report: holds a sub-report for each validation phase.*/
    private ValidationReport report;

    /** The context of this application. */
    private final ValidatorSetup setup;
    
    /**Progress bar displayed values at the end of each validation phase.*/
    private Map<ValidationPhase, Integer> percent;
    
    /**
     * List of files seen by this validator application, either because
     * included in this release structure or present in this release directory.
     */
    private final ArrayList<ValidatorFile> validatedFiles;

    public ValidatorManagerImp(final ValidatorSetup setup)
    {
        this.setup = setup;
        this.releaseParser = setup.getReleaseParser();
        this.report = new ValidationReport("Uninitialised Report", null);
        this.report.addWarn("Validation report not yet generated.");
        percent = ValidationPhase.getPercentValues(setup.isNewRelease());
        validatedFiles = new ArrayList<ValidatorFile>();
        archived = Collections.emptySet();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorManager#getErrors()
     */
    @Override
    public String getErrors()
    {
        return report.getError();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorManager#getReport()
     */
    @Override
    public ValidationReport getReport()
    {
        return report;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorManager#reportTable()
     */
    @Override
    public String reportTable()
    {
        final ValidatorToc toc = new ValidatorToc(releaseParser,
                validatedFiles, archived,
                setup.getArchivedFilesClient().hasError());
        return toc.generate();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorManager#statistic()
     */
    @Override
    public String statistic()
    {
        return setup.getValidatorStat().toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorManager#validateRelease()
     */
    @Override
    public boolean validateRelease(boolean internal)
    {
        final ArrayList<ValidationReport> subReports = new ArrayList<ValidationReport>();

        final ValidationReport initReport = new ValidationReport(
                "Check validation startup.", null);

        final ValidationReport parserReport = new ValidationReport(
                "Validation of parsing of release's structure.", null);

        final ValidationReport structureConsistencyReport = new ValidationReport(
                "Validation of Release's structure consistency.", null);

        final ValidationReport fileReport = new ValidationReport(
                "Validation of Release's disk files.", null);

        subReports.add(initReport);
        subReports.add(parserReport);
        subReports.add(structureConsistencyReport);
        subReports.add(fileReport);

        report = new ValidationReport(
                "Validation of Release rooted at base directory = "
                + setup.getConfiguration().getOptionValue(
                        ValidatorConfiguration.RELEASE_DIR.name), subReports);

        if (!validationInit(initReport))
            logger.error("Error(s) in startup. The validation process will try to continue.");
        else
            logger.debug("Setup of validation process ok.");
        setup.getProgressBar().displayPercent(percent.get(ValidationPhase.END_INIT));

        if (!buildReleaseStructure(parserReport))
            logger.error("Error(s) parsing the release's structure. The validation process will try to continue.");
        else
            logger.info("Structure of the release successfully parsed.");
        setup.getProgressBar().displayPercent(percent.get(ValidationPhase.END_RELEASE_STRUCTURE));

        logger.debug("Check structure consistency.");
        validateStructureConsistency(structureConsistencyReport);
        logger.info("Structure consistency checked.");
        setup.getProgressBar().displayPercent(percent.get(ValidationPhase.END_STRUCTURE_CONSISTENCY));
        
        logger.debug("Validating the release's disk files.");
        validateFiles(fileReport, internal);
        logger.info("Validated All the release's files on disk.");
        
        setup.getProgressBar().displayPercent(percent.get(ValidationPhase.END_VALIDATE_FILES));
        
        boolean finalResult = true;
        report.attemptStatus(report.getStatus());
        if ((report.getStatus().compareTo(STATUS.ERROR) == 0) || (report.getStatus().compareTo(STATUS.FATAL) == 0)) {
            finalResult = false;
        }
        logger.debug("The Validation Manager completed the validation with result OK="
                + finalResult);
        return finalResult;
    }

	/**
     * Return the input file IDs which are considered remotely archived by the 
     * used {@link ArchivedFilesClient}.
     * 
     * @param fileIds
     *            the file IDs to check.
     * @return list of archived file IDs.
     * @throws ValidatorException
     */
    private Set<String> getArchived(final Set<String> fileIds)
            throws ValidatorException
    {
        final ArchivedFilesClient afc = setup.getArchivedFilesClient();
        if (afc.isAvailable())
        {
            return afc.getArchived(fileIds);
        }
        else
        {
            logger.error("Cannot check if input files are archived: service not available.");
            return Collections.emptySet();
        }
    }

    /**
     * Validated the files in the directory of this release. In case of UPDATE
     * release local files and remote files are considered together to check if
     * all their names are still unique.
     * 
     * @param allFilesReport
     *            the {@link ValidationReport} for this validation phase.
     */
    private void validateFiles(final ValidationReport allFilesReport, boolean internal)
    {
        final List<String>infoFilesToExclude;
        if ( setup.isNewRelease() )
        {
            infoFilesToExclude=ValidationUtil.infoFilePaths(
                    setup.getConfiguration().getOptionValue(
                            ValidatorConfiguration.RELEASE_DIR.name), 
                            Consts.CREATE_INFO_FILES);
        }
        else 
        {
            infoFilesToExclude=ValidationUtil.infoFilePaths(
                    setup.getConfiguration().getOptionValue(
                            ValidatorConfiguration.RELEASE_DIR.name), 
                            Consts.UPDATE_INFO_FILES);
        }
        
        final Set<String> filesFromReleaseStructure = releaseParser.getCategoryMap().keySet();
        final Map<String, String> filesAreadyInError = releaseParser.getFitsInErrorMap();
        final double allFilesValidationPercentDelta = 
                percent.get(ValidationPhase.END_VALIDATE_FILES)
                - percent.get(ValidationPhase.END_STRUCTURE_CONSISTENCY);
        logger.debug("Available progress bar for validateFiles % = " 
                + allFilesValidationPercentDelta);
        final double quantum = allFilesValidationPercentDelta/10.0;
        final double incFileRelStruct = quantum/(1.0 + filesFromReleaseStructure.size());
        final Set<String> filesFromDisk = new HashSet<String>();
        // nameFiles contains only real distinct disk files, it is needed to 
        // report duplications in filesFromDisk.
        final Map<String, String> nameFiles = new HashMap<String, String>(); 

        logger.debug("Retrieved the list of disk files (" 
                + releaseParser.getLocalFilesMap().keySet().size() 
                + " files in total). Checking for duplications.");
        for (String fname : releaseParser.getLocalFilesMap().keySet())
        {
            setup.getProgressBar().increment(
                    quantum/releaseParser.getLocalFilesMap().keySet().size());
            File f = new File(releaseParser.getLocalFilesMap().get(fname));
            if (! ValidationUtil.isFileToValidate(f, infoFilesToExclude))
            {
                logger.debug("File "+ fname 
                        +" is used as release info - no validation needed.");
                continue;
            }
            
            if (!filesFromDisk.add(f.getName()))
            {
                final String msg = f.getName() + ": duplicated on disk - "
                        + nameFiles.get(f.getName()) + " , " + f.getAbsolutePath();
                logger.error(msg);
                allFilesReport.attemptStatus(STATUS.ERROR, msg);
            }
            else if (releaseParser.isRemoteFile(f.getName()))
            {
                setup.getValidatorStat().add(StatType.ERROR_DUPLICATION);
                final String msg = f.getName()
                        + ": duplicated. It's a remote file but also on disk at "
                        + f.getAbsolutePath();
                // Put in the nameFiles Map anyway because this is the first local entry.
                nameFiles.put(f.getName(), f.getAbsolutePath());
                logger.error(msg);
                allFilesReport.attemptStatus(STATUS.ERROR, msg);
            }
            else
            {
                nameFiles.put(f.getName(), f.getAbsolutePath());
            }
        }

        final double incFileDisk = 8 * (quantum / (1.0 + filesFromDisk.size()));
        logger.debug("Processing retrieved list of disk files.");
        ValidationStepRunnerManager vsrm = setup.createStepRunnerManager(incFileDisk);
        vsrm.start();
        logger.info("The ValidationStepRunnerManager was started.");
        for (String fname : filesFromDisk)
        {
            final File f = new File(nameFiles.get(fname));
            if (!filesFromReleaseStructure.contains(fname))
            {
                setup.getValidatorStat().add(StatType.ERROR_NOT_IN_RELEASE);
                final ValidatorFile vf = ValidatorFile.fileOnlyOnDisk(f);
                validatedFiles.add(vf);
                allFilesReport.attemptStatus(STATUS.ERROR, vf.getError()
                        + " (file: " + f.getAbsolutePath() + ")");
                logger.error(
                        "File only on disk and not mentioned in the release structure ("
                        + fname + ")");
                setup.getProgressBar().increment(incFileDisk);
                continue;
            }

            final String category = releaseParser.getCategory(fname);
            
            if (category == null)
            {
                setup.getValidatorStat().add(StatType.ERROR_UNPARSED_CATG);
            }
            else
            {
                setup.getValidatorStat().addCatgStat(category);
                if (!setup.getHttpConf().isValid(category))
                {
                    // Invalid categories are added also as errors (not
                    // regarding if they are added as "Others" or not) :
                    logger.error("File "+ fname+ " has an invalid category: "+ category);
                    setup.getValidatorStat().add(StatType.ERROR_INVALID_CATG);
                }
            }

            final ValidatorFile validatorFile = ValidatorFile.fileReleaseDisk(category, f);
            validatedFiles.add(validatorFile);
            boolean inError = filesAreadyInError.containsKey(fname);
            final ValidationStep vs = setup.createValidationStep(f, category, inError, internal);
            validatorFile.setValidationStep(vs);
            vsrm.enqueue(vs);
        } 
        vsrm.stop();
        logger.info("Waiting for ValidationStepRunnerManager to finish.");
        while (! vsrm.queueIsEmpty())
        {
            try
            {
                ValidationStep vs = vsrm.dequeue();
                if (vs == null)
                {
                    logger.info("Sleeping while waiting for ValidationStepRunnerManager to finish.");
                    Thread.sleep(10);
                    continue;
                }
                logger.debug(
                        "Adding to allFilesReport the report returned by ValidationStepRunnerManager for "
                        +vs.toString());
                allFilesReport.addsubreport(vs.validationReport());
            }
            catch( InterruptedException e )
            {
                logger.warn(e.toString());
            }
        }
        logger.info("ValidationStepRunnerManager finished processing all the steps.");
        
        logger.debug("Checking that all the files from the release structure are either on disk or remote.");
        setup.getProgressBar().blink();
        for (String fname : filesFromReleaseStructure)
        {
            setup.getProgressBar().increment(incFileRelStruct);
            logger.debug("Checking file from release structure: " + fname);
            if (releaseParser.isRemoteFile(fname))
            {
                logger.debug("File from release structure is from original release ["
                        + fname + "]");
                final ValidatorFile validatorFile = 
                        ValidatorFile.fileInOriginalReleaseStructure(fname, 
                                releaseParser.getCategory(fname));
                validatorFile.setValidationStep(NullValidator.validStep);
                validatedFiles.add(validatorFile);
                continue;
            }
            if (!filesFromDisk.contains(fname))
            {
                logger.error("File in release structure not present on disk: "
                        + fname);
                setup.getValidatorStat().add(StatType.ERROR_MISSING_FROM_DISK);
                final ValidatorFile vf = ValidatorFile.fileOnlyInReleaseStructure(
                        fname, releaseParser.getCategory(fname));
                validatedFiles.add(vf);
                allFilesReport.attemptStatus(STATUS.ERROR, vf.getError());
            }
            else
            {
                logger.debug("File from release structure is on disk: " + fname);
            }
        }
        setup.getProgressBar().displayPercent(
                percent.get(ValidationPhase.END_VALIDATE_FILES));
    }

    /**
     * Build the structure of the release as parsed from the fits headers. Add
     * to the statistics the found associations, split in datasets and
     * provenance: note that this is only the total number, the associations are
     * not validated yet.
     * 
     * @param relStructReport
     *            the report for this validation phase.
     * @return whether the validation was successful (true) or not (false).
     */
    private boolean buildReleaseStructure(final ValidationReport relStructReport)
    {
        boolean resOk = true;
        try
        {
            releaseParser.parse();
            relStructReport.addInfo("Release structure successfully parsed. Found defined: "
                    + releaseParser.getCategoryMap().keySet().size()
                    + " file(s) with category, "
                    + releaseParser.getDatasets().size()
                    + " dataset(s), "
                    + +releaseParser.getProvenanceMap().size()
                    + " provenance of file(s).");
        }
        catch( final ParseException e)
        {
            resOk = false;
            relStructReport.attemptStatus(STATUS.ERROR, e.getMessage());
            logger.error(e.toString());
            Map<String, String> fitsFilesInError = releaseParser.getFitsInErrorMap();
            if (fitsFilesInError != null)
            {
                Iterator<String> fitsfileNames = fitsFilesInError.keySet().iterator();
                while (fitsfileNames.hasNext()) 
                {
                    String fitsfileName = fitsfileNames.next();
                    logger.debug("Adding one to number of errors of "
                            + StatType.ERROR_INCONSISTENCY.toString() 
                            + " for parsing error of fits file: " + fitsfileName);
                    setup.getValidatorStat().add(StatType.ERROR_INCONSISTENCY);
                }
            }
        } catch (Throwable e) {
        	e.printStackTrace();
        }
        finally
        {
            logger.debug("end of parsing of the release structure.");
            setup.getProgressBar().displayPercent(percent.get(ValidationPhase.END_PARSING));
        }

        double quantum = ((double)(percent.get(ValidationPhase.END_RELEASE_STRUCTURE) 
                - setup.getProgressBar().getLastPercent())) / 3.0;
        if (quantum < 0.0)
        {
            quantum = 0.0;
        }

        logger.debug("Set statistic for remote files.");
        for (final String remoteFile : releaseParser.getRemoteFiles())
        {
            if (releaseParser.getRemovedFiles().contains(remoteFile))
            {
                continue;
            }
            final String catg = releaseParser.getCategory(remoteFile);
            setup.getValidatorStat().addCatgStat(catg);
        }
        setup.getProgressBar().increment(quantum);

        logger.debug("Set statistic for datasets.");
        final Iterator<Set<String>> dsIt = releaseParser.getDatasets().iterator();
        while( dsIt.hasNext() )
        {
            // DFS09690 SCIENCEDATASET
            // if (!dsIt.next().isEmpty())
            if (dsIt.next().size() > 1)
            {
                setup.getValidatorStat().add(StatType.DEFINED_DATASET);
            }
        }
        setup.getProgressBar().increment(quantum);
        
        logger.debug("Set statistic for provenance.");
        final Map<String, Set<String>> provenanceMap = releaseParser.getProvenanceMap();
        for (final String key : provenanceMap.keySet())
        {
            if (!provenanceMap.get(key).isEmpty())
            {
                setup.getValidatorStat().add(StatType.DEFINED_PROVENANCE);
            }
        }
        return resOk;
    }
    
    /**Check that after an update all the components of all the datasets are 
     * still part of the release. When a science file is deleted there is the 
     * risk that its dataset's components are not deleted at the same time and 
     * therefore they might be left "dangling" (without a dataset leader). 
     * @param progressBarIncrement increment of the progress bar for this check. 
     * @param structureConsistencyReport ValidationReport where to store the 
     * result of this check.
     */
    private void checkDanglingComponents(double progressBarIncrement, 
            ValidationReport structureConsistencyReport)
    {
        logger.debug(
                "Check that all the files declared as datasets' components are still available.");
        final Set<String> usedDatasetEntities = new HashSet<String>();
        for (final Set<String> datasetEntity : releaseParser.getDatasets())
        {
            usedDatasetEntities.addAll(datasetEntity);
        }
        final Set<String> remoteFilesNoLongerInDataset =  new HashSet<String>();
        remoteFilesNoLongerInDataset.addAll(releaseParser.getRemoteFiles());
        setup.getProgressBar().increment((progressBarIncrement) / 3.0);
        for (final String dsEntity : usedDatasetEntities)
        {
            remoteFilesNoLongerInDataset.remove(dsEntity);
            if (releaseParser.getRemovedFiles().contains(dsEntity))
            {
                if (!releaseParser.getLocalFilesMap().containsKey(dsEntity))
                {
                    setup.getValidatorStat().add(
                            StatType.ERROR_INCONSISTENCY);
                    final String msg = dsEntity
                            + ": was removed from the release but it is still part of a dataset definition.";
                    logger.error(msg);
                    final ValidatorFile vf = ValidatorFile.fileOnlyInReleaseStructure(
                            dsEntity, releaseParser.getCategory(dsEntity));
                    validatedFiles.add(vf);
                    structureConsistencyReport.attemptStatus(STATUS.ERROR,
                            msg);
                }
            }
        }
        logger.debug("There are "+ remoteFilesNoLongerInDataset.size() 
                + " remote files no longer in any dataset.");
        setup.getProgressBar().increment((progressBarIncrement) / 3.0);
        
        // This Set implements DFS09679:
        final Set<String> potentiallyDanglingSet = new HashSet<String>(remoteFilesNoLongerInDataset);
        for(String replacedOrRemoved: releaseParser.getRemovedFiles())
        {
            // If an ancillary file is replaced while its dataset leader is 
            // deleted, then the ancillary might remain without dataset,
            // therefore the following files are added to the set:
            if (releaseParser.getCategoryMap().containsKey(replacedOrRemoved))
            {
                logger.debug("The file was replaced, it needs to undergo dataset check: "
                        + replacedOrRemoved);
                potentiallyDanglingSet.add(replacedOrRemoved);
            }
        }
        
        logger.debug(
                "Check that all non-scientific files are still included in at least a dataset.");
        for (final String potentiallyDangling: potentiallyDanglingSet)
        {
            // This loop implements DFS09570.
            String category = releaseParser.getCategory(potentiallyDangling);
            logger.debug("Checking file (with catg="+category+"): " 
                    + potentiallyDangling);
            if (category != null) 
            { 
                if (Util.isScience(category))
                {
                    // A single science file is accepted without having to 
                    // define a dataset with itself as only component.
                    logger.debug(
                            "It is a science file, does not need to be in a dataset.");
                    continue;
                }
                
                // TODO: catalog files as well? if category.startsWith(Consts.CATALOG_SOMETHING) continue;
            }

            if (usedDatasetEntities.contains(potentiallyDangling))
            {
                logger.debug("OK - file still contained in at least a dataset.");
                continue;
            }
            else 
            {
                final String msg = potentiallyDangling
                        + ": is no longer contained in any dataset definition.";
                logger.error(msg);
                setup.getValidatorStat().add(StatType.ERROR_INCONSISTENCY);
                structureConsistencyReport.attemptStatus(STATUS.ERROR, msg);
            }
        }
        setup.getProgressBar().increment((progressBarIncrement) / 3.0);
    }
    
    /**
     * Check the consistency of the associations found in the release structure.
     * The associations are consistent when: 
     * <ul><li> All the files mentioned in the structure are available either on 
     * disk or remotely at ESO.</li>
     * <li>Provenance definitions are not circular.</li>
     * <li>Non-scientific files belong to at least a dataset.</li>
     * </ul>
     * @param structureConsistencyReport report where the results are written.
     */
    private void validateStructureConsistency(
            final ValidationReport structureConsistencyReport)
    {
        final double progressBarAvailablePercent = 
                percent.get(ValidationPhase.END_STRUCTURE_CONSISTENCY)
                - percent.get(ValidationPhase.END_RELEASE_STRUCTURE);

        final double percentIncCheckArchived = progressBarAvailablePercent / 9.0;
        final double percentIncDataset = (4.0* progressBarAvailablePercent) / 9.0;
        double percentIncProvenance = (4.0* progressBarAvailablePercent) / 9.0;

        final int percentBeforeCheckArchived = 
                percent.get(ValidationPhase.END_STRUCTURE_CONSISTENCY)
                - (int)percentIncCheckArchived;

        if (releaseParser.getRemovedFiles().isEmpty())
        {
            logger.debug(
                    "No file was deleted: there is no need to do a consistency check for the remote datasets.");
            percentIncProvenance += percentIncDataset;
        }
        else
        {
            checkDanglingComponents(percentIncDataset, structureConsistencyReport);
        }
        
        // Check that there are no loops in provenance definitions:
        final Map<String, Set<String>> provenanceMap = releaseParser.getProvenanceMap();
        final Set<String> usedProvenanceEntities = new HashSet<String>();
        for (final String key : provenanceMap.keySet())
        {
            final Set<String> set = provenanceMap.get(key);
            usedProvenanceEntities.addAll(set);
        }
        setup.getProgressBar().increment((percentIncProvenance) / 3.0);

        logger.debug("Check for circular provenance.");
        Set<String> provenanceLoops = ValidationUtil.circularDefinitions(provenanceMap);
        if (!provenanceLoops.isEmpty()) 
        {
            String msg = "Found loop(s) in provenance definition for: " 
                    + Arrays.toString(provenanceLoops.toArray());
            structureConsistencyReport.attemptStatus(STATUS.ERROR, msg);
            setup.getValidatorStat().add(StatType.ERROR_PROVENANCE);
            logger.error(msg);
        }
        
        // Check that all the provenance components exist.
        final Set<String> releaseFiles = new HashSet<String>();//local or remote.
        releaseFiles.addAll(releaseParser.getLocalFilesMap().keySet());
        releaseFiles.addAll(releaseParser.getRemoteFiles());
        setup.getProgressBar().increment((percentIncProvenance) / 3.0);
        final Set<String> provenanceComponentNotInRelease = new HashSet<String>();
        for (final String key : usedProvenanceEntities)
        {
            if (!releaseFiles.contains(key))
            {
                if (releaseParser.getRemovedFiles().contains(key))
                {
                    setup.getValidatorStat().add(StatType.ERROR_INCONSISTENCY);
                    final String msg = key
                            + ": was removed from the release but it is still part of a provenance definition.";
                    structureConsistencyReport.attemptStatus(STATUS.ERROR, msg);
                    logger.error(msg);
                }
                else
                {
                    provenanceComponentNotInRelease.add(key);
                }
            }
        }
        setup.getProgressBar().increment((percentIncProvenance) / 3.0);
        
        logger.debug("(Check of archived provenance components - currently an empty step).");
        setup.getProgressBar().displayPercent(percentBeforeCheckArchived);
        try
        {
            archived = getArchived(provenanceComponentNotInRelease);
        }
        catch( final ValidatorException e )
        {
            structureConsistencyReport.attemptStatus(STATUS.ERROR,
                    e.getMessage());
            logger.error(e.toString());
        }

        if (archived.size() < provenanceComponentNotInRelease.size())
        {
            final int missing = provenanceComponentNotInRelease.size()
                    - archived.size();
            String msg = "There are " + missing + " non local provenance files";
            if (setup.getArchivedFilesClient().hasError())
            {
                msg += " and the connection to ESO archive is not available, they will be considered not archived.";
            }
            else
            {
                msg += " and they are not archived at ESO.";
            }
            logger.error(msg);
            structureConsistencyReport.attemptStatus(STATUS.ERROR, msg);
            for (final String provenanceComponent : provenanceComponentNotInRelease)
            {
                if (!archived.contains(provenanceComponent))
                {
                    setup.getValidatorStat().add(StatType.ERROR_PROVENANCE);
                    logger.error(
                            "provenance component is not part of the release nor archived at ESO: "
                            + provenanceComponent);
                    final String catgForNotFound = releaseParser.getCategory(
                            provenanceComponent);
                    final ValidatorFile vf = ValidatorFile.fileOnlyInReleaseStructure(
                            provenanceComponent, catgForNotFound);
                    validatedFiles.add(vf);
                    structureConsistencyReport.attemptStatus(STATUS.ERROR,
                            vf.getError());
                }
            }
        }
        else
        {
            logger.debug(
                    "OK - all provenace components which are not release files are archived remotely.");
        }
        checkNotEmpty(structureConsistencyReport);
        setup.getProgressBar().displayPercent(
                percent.get(ValidationPhase.END_STRUCTURE_CONSISTENCY));
    }
    
    /** 
     * Check that a new release is not empty and that an update release 
     * contains at least an update.
     * @param structureConsistencyReport
     */
    private void checkNotEmpty(ValidationReport structureConsistencyReport)
    {
        String  errorEmptyOrNoChangeMsg = "";
        boolean errorEmptyOrNoChange = false;
        if (setup.isNewRelease())
        {
            if (releaseParser.getLocalFilesMap().size()==0)
            {
                errorEmptyOrNoChange = true;
                errorEmptyOrNoChangeMsg = "This release does not contain any file.";
            }
            else 
            {
                // fix for DFS09679 
                // TODO: if DFS09679 is rejected the enclosing else can be deleted.
                
                /* DFS11455 */
            	boolean empty = true;
            	for (String file: releaseParser.getLocalFilesMap().keySet()) {
            		if (!Consts.CREATE_INFO_FILES.contains(file))
            			empty = false;
            	}
            	
            	if (empty) {
            			errorEmptyOrNoChange = true;
            			errorEmptyOrNoChangeMsg = 
            				"This release does not contain any file (it contains only: " 
            				+ Arrays.toString(Consts.CREATE_INFO_FILES.toArray()) 
            				+" therefore it is considered empty. )";
            	}
            }
        }
        else 
        {
            if (releaseParser.getRemovedFiles().size() == 0)
            {
                errorEmptyOrNoChange = true; // unless at least a file is added.
                errorEmptyOrNoChangeMsg = "This release does not contain any valid update.";
                Set<String> localFiles = releaseParser.getLocalFilesMap().keySet();
                for(String localFile: localFiles)
                {
                    if (! Consts.UPDATE_INFO_FILES.contains(localFile)) 
                    {
                        // At least a file is added.
                        errorEmptyOrNoChange = false;
                        errorEmptyOrNoChangeMsg = "";
                        break;
                    }
                }
            }
        }

        if (errorEmptyOrNoChange)
        {
            logger.error(errorEmptyOrNoChangeMsg);
            structureConsistencyReport.attemptStatus(STATUS.ERROR, errorEmptyOrNoChangeMsg);
            setup.getValidatorStat().add(StatType.ERROR_INCONSISTENCY);
        }
    }
    
    
    /**
     * Check that the following preconditions to run a validation are met:
     * <ul>
     * <li>Executable fitsverify found.</li>
     * <li>Valid categories list available through HTTP.</li>
     * <li>HTTP connection to ESO to check for archived files available.</li>
     * </ul>
     * 
     * @param initReport
     *            the report for this validation phase.
     * @return checks OK? true/false
     */
    private boolean validationInit(final ValidationReport initReport)
    {
        boolean result = true;
        try
        {
            // Check if an executable fitsverify is found:
            final FitsVerifyExecutor fve = setup.createFitsVerifyExecutor(null);
            if (fve != null && !fve.isAvailable())
            {
                result=false;
                initReport.attemptStatus(STATUS.ERROR, fve.getError());
                setup.getValidatorStat().add(StatType.ERROR_OTHER);
                logger.error(fve.getError());
            }
        }
        catch(ValidatorException e)
        {
            // This should not happen because no file is validated here.
            result = false;
            initReport.attemptStatus(STATUS.ERROR, e.getMessage());
            logger.error(e.toString());
        }

//        try
//        {
//            // Check that the category list file is available:
//            final Map<String, String> catgs = setup.getHttpConf().getDeclaredCategories();
//            logger.debug("Category list holds " + catgs.size() + " entries.");
//        }
//        catch(Exception e)
//        {
//            // No category list: release will be set in error:
//            result = false;
//            initReport.attemptStatus(STATUS.ERROR, e.getMessage());
//            logger.error(e.toString());
//        }
        
        return result;
    }
}