/**
 * 
 */
package org.eso.phase3.validator;

import java.io.File;

import org.apache.log4j.Logger;
import org.eso.oca.fits.DataTransportFormatHandler;
import org.eso.phase3.validator.Util;
import org.eso.phase3.validator.Consts.ValidationPhase;
import org.eso.phase3.validator.ValidationReport.STATUS;
import org.eso.phase3.validator.catalog.CatalogValidator;

/**
 * This class holds the objects shared among all the validator components. The
 * factory pattern is not used for all the created objects but only for those
 * which are likely to be replaced by a mockup object during testing. An object
 * can be created by the validator factory only if it implements an interface,
 * which is then the type returned by the factory.
 * 
 * @author dsforna
 * 
 */
public class ValidatorSetupImp implements ValidatorSetup
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidatorSetupImp.class);

    private ArchivedFilesClient archivedFilesClient;

    private ValidatorConfiguration config = null;

    private ValidatorFactory factory = null;

    private HttpConfMHI httpconf = null;

    private ProgressBar progressBar;

    private ReleaseParserWithExtraInfo releaseParser = null;

    private ValidatorManager validatorManager;

    private ValidatorStat validatorStat = null;

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidatorSetup#createStepRunnerManager(double)
     */
    public ValidationStepRunnerManager createStepRunnerManager(double incPerStep)
    {
        String threadsNumber = getConfiguration().getOptionValue(
                ValidatorConfiguration.THREADS.name);
        int t = Integer.parseInt(threadsNumber);
        if (t == 0)
        {
            t = Runtime.getRuntime().availableProcessors();
        }
        
        if (t == 1)
        {
            logger.info("One single thread of execution used in this run.");
            return new SingleThreadValidationStepRunnerManager(incPerStep, progressBar);
        }
        else 
        {
            logger.info("Threads of execution used in this run: " + t);
            return new MultiThreadValidationStepRunnerManager(incPerStep, progressBar, t);
        }
    }
    
    /**
     * Create a setup object using the command line input args for the
     * configuration.
     * 
     * @param args
     * @throws Exception
     */
    public ValidatorSetupImp(final String[] args) throws ValidatorException
    {
        this.config = new ValidatorConfigurationCli(args);
        try 
        { 
            createSubObjects();
        } catch (Exception e)
        {
            throw new ValidatorException(e);
        }
    }

    /**
     * Create a setup object using a previously existing configuration.
     * 
     * @param config
     * @throws Exception
     */
    public ValidatorSetupImp(final ValidatorConfiguration config)
            throws Exception
    {
        this.config = config;
        createSubObjects();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.validator.ValidatorSetup#createFitsVerifyExecutor(java
     * .lang.String)
     */
    public FitsVerifyExecutor createFitsVerifyExecutor(final String fitsFileFullPathName) throws ValidatorException
    {
    	if (config.getOptionValue(ValidatorConfiguration.FITSVERIFY.name).equals(ValidatorConfiguration.FITSVERIFY.defaultValue))
    		return null;
    	else
    		return new FitsVerifyExecutorImp(config, fitsFileFullPathName);
    }

//    /*
//     * (non-Javadoc)
//     * 
//     * @see
//     * org.eso.phase3.validator.ValidatorSetup#createValidatorOcaParser(java
//     * .lang.String, java.lang.String, int)
//     */
//    public ValidatorOcaParser createValidatorOcaParser(
//            final String fileFullPathName, final String extCategory,
//            final int extension) throws ValidatorException
//    {
//        if (fileFullPathName == null)
//        {
//            logger.error("Null input argument: fileFullPathName");
//            throw new IllegalArgumentException(
//                    "Null input argument: fileFullPathName");
//        }
//        if (extCategory == null)
//        {
//            logger.error("Null input argument: extCategory");
//            throw new IllegalArgumentException(
//                    "Null input argument: extCategory");
//        }
//        return createValidatorOcaParser(fileFullPathName, extCategory, extension, );
//    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorSetup#getArchivedFilesClient()
     */
    public ArchivedFilesClient getArchivedFilesClient()
    {
        return archivedFilesClient;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorSetup#getConfiguration()
     */
    public ValidatorConfiguration getConfiguration()
    {
        return config;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorSetup#getHttpConf()
     */
    public HttpConfMHI getHttpConf()
    {
        return httpconf;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorSetup#getProgressBar()
     */
    public ProgressBar getProgressBar()
    {
        return progressBar;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorSetup#getReleaseParser()
     */
    public ReleaseParserWithExtraInfo getReleaseParser()
    {
        return releaseParser;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorSetup#getValidatorManager()
     */
    public ValidatorManager getValidatorManager()
    {
        return validatorManager;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ValidatorSetup#getValidatorStat()
     */
    public ValidatorStat getValidatorStat()
    {
        return validatorStat;
    }

    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidatorSetup#isNewRelease()
     */
    public boolean isNewRelease() 
    {
        final String modificationType = config.getOptionValue(ValidatorConfiguration.MODIFICATION_TYPE.name);
        logger.debug("Release modification type is: " + modificationType);
        return ValidatorConfiguration.MODIFICATION_TYPE_VALUE_NEW.equals(modificationType);
    }
    
    /**
     * Encapsulate the instantiation of the validator components.
     * 
     * @throws Exception
     */
    private void createSubObjects() throws Exception
    {
        // Low level objects with no mockup replacement are not handled by the
        // factory (even if they could, to make the implementation more
        // symmetric):
        validatorStat = new ValidatorStat();
        progressBar = new ProgressBar();

        logger.debug("Creating Factory.");
        factory = new ValidatorFactory(this);

        // The parser is also a low-level object, but it is likely to be
        // replaced by mockup for testing, so let the factory take care of it:
         
        if (isNewRelease())
        {
            logger.debug("Creating Parser for a new release.");
            releaseParser = factory.createParser();
        }
        else 
        {
            logger.debug("Creating Parser for an update release.");
            releaseParser = factory.createUpdateParser();
        }
        // The parser is a low level object and does not know about the
        // setup object: so the progress bar reference and the stat reference
        // are given explicitly. Note that the stat is updated by the parser
        // only in case of UPDATE release.
        releaseParser.setProgressBar(progressBar, ValidationPhase.getPercentValues(
                isNewRelease()).get(ValidationPhase.END_PARSING));
        releaseParser.setStat(validatorStat);
        logger.debug("Creating HttpConf.");
        httpconf = factory.createHttpConf();
        logger.debug("Creating Validator Manager.");
        validatorManager = factory.createValidatorManager();
        logger.debug("Creating Archived files client.");
        archivedFilesClient = factory.createArchivedFilesClient();
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidatorSetup#createValidationStep(java.io.File, java.lang.String, boolean)
     */
    @Override
    public ValidationStep createValidationStep(File f, String category, boolean inError, boolean internal)
    {
        logger.trace("");
        final ValidationStep vs;
        if (ValidationUtil.isConsideredFits(f.getName()))
        {
            if (Util.isCatalog(category))
            {
                vs = new CatalogValidator(this, f.getAbsolutePath(), category, inError, internal);
            }
            else 
            {
                vs = new FitsValidator(this, f.getAbsolutePath(), category, inError, internal);
            }
        }
        else
        {
            vs = new NonFitsValidator(this, f.getAbsolutePath(), category, inError);
        }
        
        if (inError)
        {
            // If an error was found for this file while parsing the release's 
            // structure from the header, the file is in error not regarding 
            // how the validation goes, but note however that the pre-existing 
            // error message is not added again to the error list.
            vs.validationReport().attemptStatus(STATUS.ERROR);
        }
        return vs;
    }

    /* (non-Javadoc)
     * Will be used to run OCA rules on extension headers.
     * @see org.eso.phase3.validator.ValidatorSetup#createValidatorOcaParser(java.lang.String, java.lang.String, int, int)
     */
    @Override
    public ValidatorOcaParser createValidatorOcaParser(String fileFullPathName,
            String extCategory, DataTransportFormatHandler fh, int headerIndex, int offset)
            throws ValidatorException
    {
        if (fileFullPathName == null)
        {
            logger.error("Null input argument: fileFullPathName");
            throw new IllegalArgumentException(
                    "Null input argument: fileFullPathName");
        }
        if (extCategory == null)
        {
            logger.error("Null input argument: extCategory");
            throw new IllegalArgumentException(
                    "Null input argument: extCategory");
        }

        
        return factory.createValidatorOcaParser(fileFullPathName, extCategory, 
        		fh, headerIndex, offset);
        
    }

}
