/**
 * 
 */
package org.eso.phase3.validator;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

/**
 * An utility class for the {@link org.eso.phase3.validator.ValidatorManager} to
 * build the TOC string of a release.
 * 
 * @author dsforna
 * 
 */
public class ValidatorToc
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ValidatorToc.class);

    private final Set<String> archived;

    private final ReleaseParser releaseParser;

    private final boolean remoteConnectionError;

    private final List<ValidatorFile> validatedFiles;

    public ValidatorToc(final ReleaseParser releaseParser,
            final List<ValidatorFile> validatedFiles,
            final Set<String> archived, final boolean remoteConnectionError)
    {
        this.releaseParser = releaseParser;
        this.validatedFiles = validatedFiles;
        this.archived = archived;
        this.remoteConnectionError = remoteConnectionError;
    }

    /**
     * Put the parsed structure of the release in a tabular format and return it
     * in a string.
     * 
     * @return the string.
     */
    public String generate()
    {
        logger.trace("");
        final String firstDs = " |--dataset-----> ";
        final String seconDs = " | ";
        final String firstProvenance = " |--provenance--> ";
        final char[] chars = new char[Consts.LINE_OUTPUT_LENGTH];
        Arrays.fill(chars, '~');
        final String dsHeaderLine = String.valueOf(chars);
        Arrays.fill(chars, '=');
        final String footerLine = String.valueOf(chars);
        final Map<String, String> diskFiles = releaseParser.getLocalFilesMap();

        final StringBuilder retSB = new StringBuilder();
        final Set<String> printed = new HashSet<String>();
        final Set<String> shifted = new HashSet<String>();
        final Set<String> notInRelease = new HashSet<String>();

        for (final ValidatorFile vf : validatedFiles)
        {
            // Each dataset and provenance component of the current file is on
            // its own row:
            final StringBuilder rows = new StringBuilder();
            final String name = vf.getName();
            if ((name.trim()).equals(""))
            {
                throw new IllegalArgumentException("empty name");
            }

            if (!vf.isInReleaseStructure())
            {
                notInRelease.add(name);
                continue;
            }
            final String catg = stringCatg(name);
            // DFS09690 SCIENCEDATASET
            Set<String> dslist = releaseParser.datesetOf(name);
            final Set<String> prlist = releaseParser.provenanceOf(name);

            // DFS09690 SCIENCEDATASET
            // Dataset consisting only of the file itself, consider as it were empty:
            if (dslist.size() == 1)
            {
                dslist = Collections.emptySet();
            }
            
            if (dslist.isEmpty() && prlist.isEmpty()
                    && (!printed.contains(name)))
            {
                // File with no component: append it at the end, unless it is
                // printed as component in the meantime:
                logger.trace("table row will be added at the end for " + name);
                shifted.add(name);
                continue;
            }

            if (!printed.add(name))
            {
                if (prlist.isEmpty() && dslist.isEmpty())
                {
                    logger.trace("Skipping already prepared table row for "
                            + name);
                    continue;
                }
            }

            logger.trace("building the table row for " + name);
            rows.append(String.format("%s%n", dsHeaderLine));
            rows.append(String.format("%-60s %-20s %-30s%n", name, "", catg));
            boolean isFirstDs = true;
            for (final String ds : dslist)
            {
                if (ds.equals(name))
                {
                    continue;
                }
                printed.add(ds);
                final String dsCatg = stringCatg(ds);
                String dsString;
                if (isFirstDs)
                {
                    isFirstDs = false;
                    dsString = firstDs;
                }
                else
                {
                    dsString = seconDs;
                }
                rows.append(String.format("%-18s%-60s %-2s %-30s%n", dsString,
                        ds, "", dsCatg));
            }
            boolean firstPr = true;
            for (final String pr : prlist)
            {
                printed.add(pr);
                String prCatg = stringCatg(pr);
                String prString;
                if (firstPr)
                {
                    firstPr = false;
                    prString = firstProvenance;
                }
                else
                {
                    prString = "";
                }
                if ((prCatg.equals(Consts.UNKNOWN_VAL))
                        && (!diskFiles.containsKey(pr)))
                {

                    prCatg = catgStringForNonLocalProvenanceEntity(pr);
                }
                rows.append(String.format("%-18s%-60s %-2s %-30s%n", prString,
                        pr, "", prCatg));
            }
            // Add the dataset/file with provenance to the printed table:
            retSB.append(rows);
        } // end loop on validatedFiles

        // Add shifted files:

        final StringBuilder notInReleaseLines = new StringBuilder();
        
        for (final String sname : shifted)
        {
            if (!printed.add(sname))
            {
                logger.debug("Skipping already prepared table row for " + sname);
                continue;
            }
            final String scatg = (releaseParser.getCategory(sname) == null) 
                    ? Consts.UNKNOWN_VAL
                    : releaseParser.getCategory(sname);
            if (notInRelease.contains(sname))
            {
                notInReleaseLines.append(String.format("%-60s %-20s %-30s%n", sname, "", scatg));
            }
            else 
            {
                retSB.append(String.format("%-60s %-20s %-30s%n", sname, "", scatg));
            }
        }
        retSB.append(String.format("%s%n", footerLine));
        retSB.append(notInReleaseLines);
        
        logger.debug("TOC was generated.");
        return retSB.toString();
    }

    /**
     * Return the correct string to display according to the available status of
     * the input provenance entity.
     * 
     * @param provenanceEntity
     *            the entity (file name or file id) from a provenance
     *            definition.
     * @return the String to display
     */
    private String catgStringForNonLocalProvenanceEntity(
            final String provenanceEntity)
    {

        if (remoteConnectionError)
        {
            return Consts.MAYBE_ARCHIVED;
        }
        else if (archived.contains(provenanceEntity))
        {
            return Consts.ARCHIVED;
        }
        else
        {
            return Consts.UNKNOWN_VAL;
        }

    }

    /**
     * Return the category of a file as string that can be written in the TOC
     * file.
     * 
     * @param name
     *            name of the file.
     * @return the category as a string.
     */
    private String stringCatg(final String fileName)
    {
        final String catg = releaseParser.getCategory(fileName);
        if (catg == null)
        {
            return Consts.UNKNOWN_VAL;
        }
        return catg;
    }
}
