/**
 * 
 */
package org.eso.phase3.validator.catalog;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.eso.oca.fits.DataTransportFormatHandler;
import org.eso.oca.fits.TypedHeaderCard;
import org.eso.phase3.catalog.domain.Catalog;
import org.eso.phase3.validator.FitsValidator;
import org.eso.phase3.validator.Util;
import org.eso.phase3.validator.ValidationReport;
import org.eso.phase3.validator.ValidationStep;
import org.eso.phase3.validator.ValidationUtil;
import org.eso.phase3.validator.ValidatorSetup;
import org.eso.phase3.validator.ValidationReport.STATUS;
import org.eso.phase3.validator.ValidatorStat.StatType;

import uk.ac.starlink.fits.FitsTableBuilder;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.RowSequence;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StoragePolicy;
import uk.ac.starlink.util.FileDataSource;

/**
 * Validate this catalog fits file.  
 * @author dsforna
 *
 */
public class CatalogValidator implements ValidationStep
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(CatalogValidator.class);
    
    /* used for standard fits validation */
    private final ValidationStep fitsStep;
    
    private boolean catalogIsValid;
    private ValidationReport catalogReport;
    private ValidatorSetup setup;
    private String absolutePathName;
    private String fileName;
    private String category;
    
    private Pattern dataRangePattern = Pattern.compile("(TD[A-Z]+)(\\d+)");
    
    private Double[][]	dataRanges;
        
    /**
     * @param validatorSetupImp
     * @param absolutePath
     * @param category
     * @param inError
     */
    public CatalogValidator(ValidatorSetup setup, String absolutePathName, String category, boolean inError, boolean internal)
    {
        this.setup  = setup;
        this.absolutePathName = absolutePathName;
        this.category = category;
        
        this.fileName = (new File(absolutePathName)).getName();
        
        fitsStep = new FitsValidator(setup, absolutePathName, category, inError, internal);
        catalogIsValid = false;
        catalogReport = new ValidationReport("CatalogValidator on fits catalog "
                + absolutePathName + " [catg:" + category + "]", null);
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#isValid()
     */
    @Override
    public boolean isValid()
    {
        logger.trace("CatalogValidator::isValid");
        return (catalogIsValid && fitsStep.isValid());
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#runValidation()
     */
    @Override
    public void runValidation()
    {
        logger.trace("CatalogValidator::runValidation");
        /* fits validation */
        fitsStep.runValidation();
        /* catalog validation */
        runCatalogValidation();
        /* merge reports */
        catalogReport.addsubreport(fitsStep.validationReport());
    }

    private void checkIndexedKeywords(Map<String, TypedHeaderCard> keyCardMap, int tfields)
    {
    	IndexedKeywordsValidator kwdValidator = new IndexedKeywordsValidator(fileName, tfields, catalogReport, setup.getValidatorStat());
    	dataRanges = new Double[2][tfields];
		for (TypedHeaderCard card: keyCardMap.values())
        {
    		/* get keywords for validation */
    		kwdValidator.add(card);
    		
    		/* extract TDMIN, TDMAX  */
    		Matcher m = dataRangePattern.matcher(card.getKey());
    		if (m.matches()) {
    			int idx = Integer.valueOf(m.group(2)) - 1;
    			if (m.group(1).equals("TDMIN")) {
    				dataRanges[0][idx] = Double.valueOf(card.getValue());
    			} else if (m.group(1).equals("TDMAX")) {
    				dataRanges[1][idx] = Double.valueOf(card.getValue());
    			}
    		}
        }
		
		/* check TDMIN and TDMAX consistency */
		for (int i = 0; i < tfields; i++) {
			Double min = dataRanges[0][i];
			Double max = dataRanges[1][i];
			if (min != null && max != null) {
				if (max < min) { 
					String msg = new File(absolutePathName).getName() + ": TDMAX" + (i+1) + " is smaller than TDMIN" + (i+1);
					catalogReport.attemptStatus(STATUS.ERROR, msg);
					logger.error(msg);
					setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
				}
			}
		}
		
		/* perform the actual validation */
        kwdValidator.process();
    }
    
    private void runCatalogValidation()        
    { 
    	/* index of main header */
        final int idxMainHeader = setup.getReleaseParser().getIndexParsedHeader(fileName);
        
        DataTransportFormatHandler dtfh = null;
        try
        {
        	dtfh = ValidationUtil.allocateDTFH(absolutePathName);
        	final int idxExtensionHeader = Util.findCatalogHeader(dtfh);
            
        	int tfields = 0;
            
            /* checks on main header (UR2) */
        	TypedHeaderCard[] naxisCard = dtfh.getFITSCards(idxMainHeader, new String[]{"NAXIS"});
        	if (naxisCard.length == 1 && naxisCard[0].isDefined()) {
        		int naxis = Integer.parseInt(naxisCard[0].getValue());
        		if (naxis != 0) {
        			String msg = fileName + ": NAXIS in main header is not 0";
        			logger.error(msg);
        			catalogReport.attemptStatus(STATUS.ERROR, msg);
        			setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
        		}
        	} else {
        		String msg = fileName + ": the main header does not contain the NAXIS keyword";
        		logger.error(msg);
    			catalogReport.attemptStatus(STATUS.ERROR, msg);
    			setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
        	}
            
        	/* check that the extension header is a bintable (UR1) */
            TypedHeaderCard[] tfieldsCard = dtfh.getFITSCards(idxExtensionHeader, new String[]{"TFIELDS", "XTENSION"});
            if (tfieldsCard.length == 2 && tfieldsCard[0].isDefined() && tfieldsCard[1].isDefined()) {
                tfields = Integer.parseInt(tfieldsCard[0].getValue());
                if (!"BINTABLE".equals(tfieldsCard[1].getValue())) {
                    String msg = fileName + ": XTENSION is not set to 'BINTABLE'";
                    logger.error(msg);
        			catalogReport.attemptStatus(STATUS.ERROR, msg);
        			setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
                }
            } else {
                String msg = fileName + ": not all the requested keywords (TFIELDS, XTENSION) are available.";
                logger.error(msg);
    			catalogReport.attemptStatus(STATUS.ERROR, msg);
    			setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
            }

            /* check conditions on single keywords */
            TypedHeaderCard[] cards = dtfh.getFITSCards(idxExtensionHeader);
            Map<String, TypedHeaderCard> keyCardMap = new HashMap<String, TypedHeaderCard>();
            for (TypedHeaderCard card: cards)
            	keyCardMap.put(card.getKey(), card);
            
            checkIndexedKeywords(keyCardMap, tfields);
            
            /* 
             * try to create the Catalog object (this will be done later by ExtractCatalog
             * with some hard-coded logic, therefore it should be tested also here
             */
            boolean mainFile = !Util.isBigCatalogData(category);
            Catalog.fromFitsFile(absolutePathName, mainFile);
            
            checkData(new File(absolutePathName));
        } catch (Throwable e) {
        	String msg = new File(absolutePathName).getName() + ": error reading extension header: "
            				+ e.getMessage();
            logger.error(msg);
            catalogReport.attemptStatus(STATUS.ERROR, msg); 
            setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
        } finally {
        	catalogIsValid = catalogReport.getStatus().equals(STATUS.VALID);
        	if (dtfh != null)
        		dtfh.dispose();
        }
    }

	private void checkData(File catalog) throws IOException {
    	FitsTableBuilder ftb = new FitsTableBuilder();
    	/* create star table */
    	StarTable table = ftb.makeStarTable(new FileDataSource(catalog.getAbsolutePath()), false, StoragePolicy.getDefaultPolicy());

    	int ncol = table.getColumnCount();
    	int identifierIdx = -1;
    	Set<Object> identifierValues = new TreeSet<Object>();
    	
    	Class<?>[] types = new Class[ncol];
    	for (int i = 0; i < ncol; i++) {
    		ColumnInfo colInfo = table.getColumnInfo(i);
    		types[i] = colInfo.getContentClass();
    		if (TUcdNKwdValidator.UNIQUE_ID_UCD.equals(colInfo.getUCD()))
    			identifierIdx = i;
    	}
    	
    	int rowIndex = -1;
    	RowSequence rseq = table.getRowSequence();
    	while (rseq.next()) {
    		rowIndex +=1;
    		Object[] row = rseq.getRow();
    		for (int i = 0; i < table.getColumnCount(); i++) {
    			/* check for null data */
    			if (i == identifierIdx) {
    				if (row[i] == null) {
    					String msg = catalog.getName() + ": catalog data in identifier column " + i + ", row " + rowIndex + " is null, data check will not continue";
    					logger.error(msg);
    					catalogReport.attemptStatus(STATUS.ERROR, msg);
    					setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
    					return;
    				} else if (identifierValues.contains(row[i])){
    					String msg = catalog.getName() + ": identifier column contains a duplicated value [" + row[i] + "] in row " + rowIndex + ", data check will not continue";
    					logger.error(msg);
    					catalogReport.attemptStatus(STATUS.ERROR, msg);
    					setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
    					return;
    				} else {
    					identifierValues.add(row[i]);
    				}
    			}
    			/* check data ranges */
    			if (Number.class.isAssignableFrom(types[i])) {
    				if (row[i] == null) 
    					continue;
    				double val = ((Number) row[i]).doubleValue();
    				/* DFS11672 */
    				if (Double.isInfinite(val)) {
    					String msg = catalog.getName() + ": catalog data in row " + rowIndex + ", column " + i + " contains an infinite value, which is not allowed. Data check will not continue";
    					logger.error(msg);
    					catalogReport.attemptStatus(STATUS.ERROR, msg);
    					setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
    					return;
    				} else if (dataRanges[0][i] != null && val < dataRanges[0][i]) {
    					String msg = catalog.getName() + ": catalog data in row " + rowIndex + ", column " + i + " is out of min range " + dataRanges[0][i] + ", data check will not continue";
    					logger.error(msg);
    					catalogReport.attemptStatus(STATUS.ERROR, msg);
    					setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
    					return;
    				} else if (dataRanges[1][i] != null && val > dataRanges[1][i]) {
    					String msg = catalog.getName() + ": catalog data in row " + rowIndex + ", column " + i + " is out of max range " + dataRanges[1][i] + ", data check will not continue";
    					logger.error(msg);
    					catalogReport.attemptStatus(STATUS.ERROR, msg);
    					setup.getValidatorStat().add(StatType.ERROR_CATALOG_VALIDATION);
    					return;
    				}
    			}
    			/* check null strings */
    			else if (String.class.isAssignableFrom(types[i])) {
    				if (CatalogConsts.NULL_STRING.equals(row[i]))
    					row[i] = null;
    			}
    		}
    	}
    	rseq.close();
    	
    }

	/* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#validationReport()
     */
    @Override
    public  synchronized  ValidationReport validationReport()
    {
        logger.trace("CatalogValidator::validationReport");
        return catalogReport;
    }
}
