package org.eso.phase3.validator.catalog;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.eso.oca.fits.TypedHeaderCard;
import org.eso.phase3.validator.ValidationReport;
import org.eso.phase3.validator.ValidationReport.STATUS;
import org.eso.phase3.validator.ValidationUtil;
import org.eso.phase3.validator.ValidatorStat;
import org.eso.phase3.validator.ValidatorStat.StatType;

public class TTypeNKwdValidator extends IndexedKeywordValidator {
	
	/** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(TFormNKwdValidator.class);
    
    public static final String kwIdentifier = "TTYPE";
    
    public static final String SQL_RESERVED_WORDS_RESOURCE="sql_reserved_words.txt";
    private static Set<String> reservedWords = new HashSet<String>();
    
    private Set<String> foundColumnNames = new TreeSet<String>(new Comparator<String>() {
		@Override
		public int compare(String o1, String o2) {
			return o1.compareToIgnoreCase(o2);
		}
    });
    
    static {
        try {
            reservedWords = ValidationUtil.readAsSet(ClassLoader.getSystemResourceAsStream(SQL_RESERVED_WORDS_RESOURCE));
        } catch (Exception e) {
            logger.error("Error loading the list of sql reserved words from resource "
                    + SQL_RESERVED_WORDS_RESOURCE + " - " + e.toString());
            reservedWords = Collections.emptySet();
        }
    }
    
    // Values for TTYPEn keyword must be in this format [see DMS 6.1.1.1]:
    private static Pattern identifierPattern = Pattern.compile(CatalogConsts.identifierRegExp);
	
	public TTypeNKwdValidator(int tFields, ValidationReport report, ValidatorStat stat, String fileName) {
		super(tFields, report, stat, fileName);
	}
	
	@Override
	public void process() {
		for (TypedHeaderCard card: cards) {
			String columnName = card.getValue();
			/* check for reserved words */
			if (reservedWords.contains(columnName.toLowerCase())) {
				String msg = "[" + fileName + "] - Reserved word as column name (" + columnName +") - keyword:" + card.getKey();
				logger.error(msg);
				report.attemptStatus(STATUS.ERROR, msg);
				stat.add(StatType.ERROR_CATALOG_VALIDATION);
			/* check for duplicates */
			} else if (foundColumnNames.contains(columnName)) {
				String msg = "[" + fileName + "] - Duplicated column name (" + columnName +") - keyword:" + card.getKey();
				logger.error(msg);
				report.attemptStatus(STATUS.ERROR, msg);
				stat.add(StatType.ERROR_CATALOG_VALIDATION);
			/* check format */
			} else if (identifierPattern.matcher(columnName).matches()) {
				foundColumnNames.add(columnName);
			} else {
				String msg = "[" + fileName + "] - Column name (" + columnName +") does not match the identifier patter - keyword:" + card.getKey();
				logger.error(msg);
				report.attemptStatus(STATUS.ERROR, msg);
				stat.add(StatType.ERROR_CATALOG_VALIDATION);
			}
		}
		
		if (cards.size() != totalFields) {
			String msg = "[" + fileName + "] - Number of TTYPE keywords does not match the total number of fields";
			logger.error(msg);
			report.attemptStatus(STATUS.ERROR, msg);
			stat.add(StatType.ERROR_CATALOG_VALIDATION);
		}
	}

}
