package org.eso.phase3.validator.catalog;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.eso.oca.fits.TypedHeaderCard;
import org.eso.phase3.validator.ValidationReport;
import org.eso.phase3.validator.ValidationUtil;
import org.eso.phase3.validator.ValidatorStat;
import org.eso.phase3.validator.ValidationReport.STATUS;
import org.eso.phase3.validator.ValidatorStat.StatType;

public class TUcdNKwdValidator extends IndexedKeywordValidator {
	
	/** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(TFormNKwdValidator.class);
    
    public static final String kwIdentifier = "TUCD";
    
    public static final String UCD1PLUS_RESOURCE = "ucdplus_V1.23_02APR2007.txt";
    public static final String UNIQUE_ID_UCD = "meta.id;meta.main";
    
    private int ucdUniqueIdCount = 0;    
    
    private static Set<String> validUcd1Plus = new HashSet<String>();
    
    static {
        try {
            validUcd1Plus = ValidationUtil.readAsSet(ClassLoader.getSystemResourceAsStream(UCD1PLUS_RESOURCE));
        } catch( Exception e) {
            logger.error("Error loading the list of UCD1+ from resource "
                    + UCD1PLUS_RESOURCE + " - " + e.toString());
            validUcd1Plus = Collections.emptySet();
        }
    }
    
    // Values for TTYPEn keyword must be in this format [see DMS 6.1.1.1]:
    Pattern ttypePattern = Pattern.compile("[A-Z][A-Z_0-9]*");
	
	public TUcdNKwdValidator(int tFields, ValidationReport report, ValidatorStat stat, String fileName) {
		super(tFields, report, stat, fileName);
	}
	
	@Override
	public void process() {
		for (TypedHeaderCard card: cards) {
			String fullUcd = card.getValue();
			if (UNIQUE_ID_UCD.equals(fullUcd)) {
				ucdUniqueIdCount++;
				if (ucdUniqueIdCount > 1) {
					String msg = "[" + fileName + "] - Repeated column with unique ID ucd = " + fullUcd +" - keyword:" + card.getKey();
					logger.error(msg);
					report.attemptStatus(STATUS.ERROR, msg);
					stat.add(StatType.ERROR_CATALOG_VALIDATION);
				}
			}
			String[] ucdAtoms = fullUcd.split(";"); 
			for (String ucd : ucdAtoms)
				if (! validUcd1Plus.contains(ucd)) {
					String msg = "[" + fileName + "] - Invalid ucd component (" + ucd + ") in ucd =" + fullUcd + " - keyword:" + card.getKey();
					logger.error(msg);
					report.attemptStatus(STATUS.ERROR, msg);
					stat.add(StatType.ERROR_CATALOG_VALIDATION);
				}
		}

		if (ucdUniqueIdCount == 0) {
			String msg = "[" + fileName + "] - Missing column with unique ID ucd=" + UNIQUE_ID_UCD;
			logger.error(msg);
			report.attemptStatus(STATUS.ERROR, msg);
			stat.add(StatType.ERROR_CATALOG_VALIDATION);
		}	
	}

}
