/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/

package org.openorb.compiler.idl.util;

import org.openorb.compiler.object.IdlObject;
import org.openorb.compiler.object.IdlComment;
import org.openorb.compiler.object.IdlCommentSection;
import org.openorb.compiler.object.IdlCommentField;
import org.openorb.compiler.parser.IdlType;

/**
 * This class provides several tools to help user to define its own translation.
 *
 * @author Jerome Daniel
 * @version $Revision: 1.5 $ $Date: 2002/08/21 08:19:21 $ 
 */

public class tools
{

    public static final String tab = "    ";

    /**
     * Reference to the generator
     */
    protected static org.openorb.compiler.generator.IdlToJava generator = new org.openorb.compiler.generator.IdlToJava();

    /**
     * This operation creates and returns a print access to a new file.
     */
    public static java.io.PrintWriter createNewFile( java.io.File into, String name )
    {
        String path;
        java.io.PrintWriter printout = null;

        path = new String( into.getPath() + java.io.File.separator + name + ".java" );

        java.io.File file = new java.io.File( path );

        try
        {
            java.io.FileOutputStream output = new java.io.FileOutputStream( file );
            java.io.DataOutputStream dataout = new java.io.DataOutputStream( output );
            printout = new java.io.PrintWriter( dataout );
        }
        catch ( java.io.IOException e )
        {
            e.printStackTrace();
        }

        return printout;
    }

    /**
     * This operation prints the object package name ( according to the IDL to Java translation rules ).
     */
    public static void writePackageName( java.io.PrintWriter writer, org.openorb.compiler.idl.reflect.idlObject obj )
    {
        if ( obj.idlDefinedIn() != null )
            writer.println( "package " + fullname( obj.idlDefinedIn() ) + ";" );
    }

    /**
     * This operation returns a Java name for an IDL object accroding to the IDL to Java mapping rules.
     */
    public static String javaName( org.openorb.compiler.idl.reflect.idlObject obj )
    {
        if ( obj.idlType() == org.openorb.compiler.idl.reflect.idlType.IDENTIFIER )
            return fullname( ( ( org.openorb.compiler.idl.reflect.idlIdentifier ) obj ).original() );

        return fullname( obj );
    }

    private static String fullname ( org.openorb.compiler.idl.reflect.idlObject obj )
    {
        java.util.Vector v = new java.util.Vector();
        org.openorb.compiler.object.IdlObject obj2 = ( org.openorb.compiler.object.IdlObject ) obj;
        String name = new String( "" );
        String s;
        boolean first = false;

        while ( obj2 != null )
        {
            if ( first )
            {
                if ( ( obj2.kind() == org.openorb.compiler.parser.IdlType.e_interface ) ||
                        ( obj2.kind() == org.openorb.compiler.parser.IdlType.e_value ) ||
                        ( obj2.kind() == org.openorb.compiler.parser.IdlType.e_struct ) ||
                        ( obj2.kind() == org.openorb.compiler.parser.IdlType.e_union ) ||
                        ( obj2.kind() == org.openorb.compiler.parser.IdlType.e_exception ) )
                    v.addElement( ( obj2.name() + "Package" ) );
                else
                    v.addElement( obj2.name() );
            }
            else
                v.addElement( obj2.name() );

            if ( obj2.upper() != null )
                if ( obj2.upper().kind() == org.openorb.compiler.parser.IdlType.e_root )
                    break;

            obj2 = obj2.upper();

            first = true;
        }

        if ( org.openorb.compiler.IdlCompiler.packageName != null )
        {
            if ( !obj.included() )
            {
                if ( !org.openorb.compiler.IdlCompiler.packageName.equals( "" ) )
                {
                    if ( !( ( org.openorb.compiler.IdlCompiler.packageName.equals( "generated" ) ) && ( org.openorb.compiler.IdlCompiler.use_package == false ) ) )
                        name = adaptToDot( org.openorb.compiler.IdlCompiler.packageName );
                }
            }
        }

        if ( org.openorb.compiler.IdlCompiler.usePrefix )
            if ( ( ( org.openorb.compiler.object.IdlObject ) obj ).getPrefix() != null )
            {
                if ( !name.equals( "" ) )
                    name = name + ".";

                name = name + inversedPrefix( ( ( org.openorb.compiler.object.IdlObject ) obj ) .getPrefix() );
            }

        for ( int i = v.size() - 1; i >= 0; i-- )
        {
            s = ( String ) v.elementAt( i );

            if ( s != null )
            {
                if ( !name.equals( "" ) )
                    name = name + ".";

                name = name + s;
            }
        }

        return name;
    }

    public static String adaptToDot( String path )
    {
        char [] tmp = new char[ path.length() ];

        for ( int i = 0; i < path.length(); i++ )
        {
            if ( ( path.charAt( i ) == '/' ) || ( path.charAt( i ) == '\\' ) )
                tmp[ i ] = '.';
            else
                tmp[ i ] = path.charAt( i );
        }

        return new String( tmp );
    }

    public static String inversedPrefix ( String prefix )
    {
        int index = 0;
        int previous_index = 0;
        java.util.Vector seq = new java.util.Vector();
        String inversed = new String( "" );

        try
        {
            while ( index != -1 )
            {
                index = prefix.indexOf( '.', previous_index );

                if ( index != -1 )
                {
                    seq.addElement( new String( prefix.substring( previous_index, index ) ) );
                    previous_index = index + 1;
                }
            }
        }
        catch ( StringIndexOutOfBoundsException ex )
        { }

        seq.addElement( new String( prefix.substring( previous_index, prefix.length() ) ) );

        for ( int i = seq.size() - 1; i >= 0; i-- )
        {
            if ( !inversed.equals( "" ) )
                inversed = inversed + ".";

            inversed = inversed + ( String ) seq.elementAt( i );
        }

        return inversed;
    }

    /**
     * This operations translates an IDL object to java
     */
    public static void javaType( org.openorb.compiler.idl.reflect.idlObject obj, java.io.PrintWriter output )
    {
        if ( obj.idlType() == org.openorb.compiler.idl.reflect.idlType.IDENTIFIER )
        {
            generator.translate_type( ( org.openorb.compiler.object.IdlObject ) ( ( org.openorb.compiler.idl.reflect.idlIdentifier ) obj ).original(), output );
            return ;
        }

        generator.translate_type( ( org.openorb.compiler.object.IdlObject ) obj, output );
    }

    /**
     * This operations marshals an IDL object to java
     */
    public static void marshal( org.openorb.compiler.idl.reflect.idlObject obj, java.io.PrintWriter output, String streamName, String typeName )
    {
        generator.translate_marshalling_member( ( org.openorb.compiler.object.IdlObject ) obj, output, streamName, typeName, tab + tab );
    }

    /**
     * This operations unmarshals an IDL object to java
     */
    public static void unmarshal( org.openorb.compiler.idl.reflect.idlObject obj, java.io.PrintWriter output, String streamName, String typeName )
    {
        generator.translate_unmarshalling_member( ( org.openorb.compiler.object.IdlObject ) obj, output, streamName, typeName, tab + tab );
    }


    /**
     * Translate a JavaDoc comments section
     */
    public static void translate_comment_section( java.io.PrintWriter output, String description, IdlObject obj )
    {
        translate_comment_section_base( output, description, obj );
    }

    /**
     * Translate a JavaDoc comments section
     */
    public static void translate_comment_section_base( java.io.PrintWriter output, String description, IdlObject obj )
    {
        int i = 0;

        while ( i < description.length() )
        {
            if ( description.charAt( i ) == '\n' )
            {
                if ( i != description.length() - 1 )
                {
                    output.println( "" );

                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) || ( obj.kind() == IdlType.e_state_member ) )
                        output.print( tab );

                    output.print( " * " );
                }
                else
                {
                    output.println( "" );
                    return ;
                }
            }
            else
                output.print( description.charAt( i ) );

            i++;
        }
    }

    /**
     * Add a JavaDoc comment
     *
     * @param output  the target file
     * @param obj   the object the header has to be added
     */
    public static void javadoc ( java.io.PrintWriter output, IdlObject obj )
    {
        javadoc_base( output, obj );
    }

    /**
     * Add a JavaDoc comment
     *
     * @param output  the target file
     * @param obj   the object the header has to be added
     */
    public static void javadoc_base ( java.io.PrintWriter output, IdlObject obj )
    {
        IdlComment comment = obj.getComment();
        String description = null;

        if ( comment != null )
        {
            description = comment.get_description();

            if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) || ( obj.kind() == IdlType.e_state_member ) )
                output.print( tab );

            output.println( "/**" );

            if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) || ( obj.kind() == IdlType.e_state_member ) )
                output.print( tab );

            output.print( " * " );

            translate_comment_section( output, description, obj );

            IdlCommentSection [] sections = comment.get_sections();

            for ( int i = 0; i < sections.length; i++ )
            {
                switch ( sections[ i ].kind().value() )
                {

                case IdlCommentField._author_field :

                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @author " );

                    break;

                case IdlCommentField._deprecated_field :
                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @deprecated " );

                    break;

                case IdlCommentField._exception_field :
                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @exception " );

                    break;

                case IdlCommentField._return_field :
                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @return " );

                    break;

                case IdlCommentField._param_field :
                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @param " );

                    break;

                case IdlCommentField._see_field :
                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @see " );

                    break;

                case IdlCommentField._version_field :
                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @version " );

                    break;

                case IdlCommentField._unknown_field :
                    if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) )
                        output.print( tab );

                    output.print( " * @" + sections[ i ].get_title() + " " );

                    break;
                }

                description = sections[ i ].get_description();
                translate_comment_section( output, description, obj );
            }

            if ( ( obj.kind() == IdlType.e_attribute ) || ( obj.kind() == IdlType.e_operation ) || ( obj.kind() == IdlType.e_state_member ) )
                output.print( tab + "" );

            output.println( " */" );
        }
    }

}
