/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/

package org.openorb.compiler.parser;

import org.openorb.compiler.object.*;

/**
 * Cette classe represente la grammaire IDL 
 *
 * @author Jerome Daniel
 * @version $Revision: 1.9 $ $Date: 2002/08/21 08:19:24 $ 
 */

public class IdlGrammar
{

    // ---------
    // Attributs
    // ---------

    /**
    * Permet d'acceder au parser pour parcourir le fichier IDL
    */
    public IdlParser parser = null;

    /**
     * Permet d'accer  l'IR importeur
     */
    //public IRImport irImport = null;

    // ------------
    // CONSTRUCTEUR
    // ------------
    /**
    * Construit une nouvelle class pour la gestion de la grammaire IDL
    */
    public IdlGrammar( IdlParser p )
    {
        parser = p;

        //irImport = loadIRImport( p );
    }

    /**
     * This operation is used to load the IR Import tool.
    public IRImport loadIRImport( IdlParser p )
    {
        IRImport ir_import = null;

        try
        {
            java.lang.Class clz = Thread.currentThread().getContextClassLoader().loadClass( "org.openorb.compiler.ir.IdlFromIR" );

            java.lang.Object obj = clz.newInstance();

            ir_import = ( IRImport ) obj;

            ir_import.set_parser( p );
        }
        catch ( java.lang.Throwable ex )
        {}

        return ir_import;
    }
     */

    // -----------
    // HEXA TO DEC
    // -----------
    /**
     * Cette fonction retourne la valeur decimale d'un symbole hexadecimal
     *
     * @param hexa le symbole hexadecimal
     * @return la valeur decimale correspondante
     */
    public int hexaToDec( char hexa )
    {
        switch ( hexa )
        {

        case '0' :
            return 0;

        case '1' :
            return 1;

        case '2' :
            return 2;

        case '3' :
            return 3;

        case '4' :
            return 4;

        case '5' :
            return 5;

        case '6' :
            return 6;

        case '7' :
            return 7;

        case '8' :
            return 8;

        case '9' :
            return 9;

        case 'a' :
            return 10;

        case 'b' :
            return 11;

        case 'c' :
            return 12;

        case 'd' :
            return 13;

        case 'e' :
            return 14;

        case 'f' :
            return 15;
        }

        return 0;
    }

    // ------------------
    // CONVERT TO DECIMAL
    // ------------------
    /**
     * Cette fonction convertie un nombre hexadecimal en nombre decimal
     * Le nombre est stocke dans ctx.value sous forme d'une chaine.
     */
    public void convertToDecimal()
    {
        String significatif = parser.ctx.value.substring( parser.ctx.value.indexOf( 'x' ) + 1 );
        int value = 0;

        for ( int i = 0; i < significatif.length(); i++ )
        {
            value = value * 16;
            value = value + hexaToDec ( significatif.charAt( i ) );
        }

        parser.ctx.value = new Integer( value ).toString();
    }

    /**
     * Inverse l'ordre d'un prefixe : omg.org -> org.omg
     *
     * @param prefix  le prefixe a inverser
     * @return le prefixe inverse
     */
    public String inversedPrefix ( String prefix )
    {
        int index = 0;
        int previous_index = 0;
        java.util.Vector seq = new java.util.Vector();
        String inversed = new String( "" );

        try
        {
            while ( index != -1 )
            {
                index = prefix.indexOf( '.', previous_index );

                if ( index != -1 )
                {
                    seq.addElement( new String( prefix.substring( previous_index, index ) ) );
                    previous_index = index + 1;
                }
            }
        }
        catch ( StringIndexOutOfBoundsException ex )
        { }

        seq.addElement( new String( prefix.substring( previous_index, prefix.length() ) ) );

        for ( int i = seq.size() - 1; i >= 0; i-- )
        {
            if ( !inversed.equals( "" ) )
                inversed = inversed + ".";

            inversed = inversed + ( String ) seq.elementAt( i );
        }

        return inversed;
    }

    /**
     * Retourne le nom complet d'un objet CORBA 
     *
     * @param obj l'objet dont on doit trouver le nom complet
     * @return le nom complet
     */
    public String fullname ( IdlObject obj )
    {
        java.util.Vector v = new java.util.Vector();
        IdlObject obj2 = obj;
        String name = new String( "" );
        String s;
        boolean first = false;

        while ( obj2 != null )
        {
            if ( first )
            {
                if ( obj2.kind() == IdlType.e_interface )
                {
                    if ( obj.kind() != IdlType.e_const )
                        v.addElement( ( obj2.name() + "Package" ) );
                    else
                        v.addElement( obj2.name() );
                }
                else
                    v.addElement( obj2.name() );
            }
            else
                v.addElement( obj2.name() );


            //if ( org.openorb.compiler.IdlCompiler.use_package == false )
            //{
            if ( obj2.upper() != null )
                if ( obj2.upper().kind() == IdlType.e_root )
                    break;

            //}

            obj2 = obj2.upper();

            first = true;
        }

        if ( org.openorb.compiler.IdlCompiler.packageName != null )
        {
            if ( !org.openorb.compiler.IdlCompiler.packageName.equals( "" ) )
            {
                if ( !( ( org.openorb.compiler.IdlCompiler.packageName.equals( "generated" ) ) && ( org.openorb.compiler.IdlCompiler.use_package == false ) ) )
                    name = org.openorb.compiler.idl.util.tools.adaptToDot( org.openorb.compiler.IdlCompiler.packageName );
            }
        }

        if ( ( obj.getPrefix() != null ) && ( org.openorb.compiler.IdlCompiler.usePrefix == true ) )
        {
            if ( !name.equals( "" ) )
                name = name + ".";

            name = name + inversedPrefix( obj.getPrefix() );
        }

        for ( int i = v.size() - 1; i >= 0; i-- )
        {
            s = ( String ) v.elementAt( i );

            if ( s != null )
            {
                if ( !name.equals( "" ) )
                    name = name + ".";

                name = name + s;
            }
        }

        return name;
    }

    /**
     * Remove the first '_' char
     */
    private String removePrefix( String name )
    {
        if ( name.charAt( 0 ) == '_' )
            return name.substring( 1 );
        else
            return name;
    }

    // ---------------
    // < SCOPED NAME >
    // ---------------
    /**
     * (11)
     * Analyse un nom eventuellement compose
     */
    public String scoped_name()
    {
        String name = new String();
        boolean stop = false;

        while ( stop == false )
        {
            if ( parser.ctx.symb == Token.t_quatre_pts )
            {
                name = name + "::";
                parser.symbole();
            }

            if ( parser.ctx.symb != Token.t_ident )
            {
                parser.show_error( "Bad identifier" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return name;
            }

            name = name + removePrefix( parser.ctx.value );

            parser.symbole();

            if ( parser.ctx.symb != Token.t_quatre_pts )
                stop = true;

        }

        return name;
    }

    // ---------------
    // < PRIMARY EXP >
    // ---------------
    /**
     * (23)
     * Analyse une expression constante
     */
    public long primary_exp( StringContainer str, IdlObject obj, DoubleContainer fv )
    {
        String name;
        IdlObject found;
        long value = 0;

        switch ( parser.ctx.symb )
        {

        case Token.t_quatre_pts :

        case Token.t_ident :
            name = scoped_name();

            if ( obj.isVisible( name, false ) == false )
            {
                parser.show_error( "Undefined identifier : " + name );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }
            else
            {
                found = obj.returnVisibleObject( name, false );

                if ( found.kind() == IdlType.e_const )
                {
                    value = ( ( IdlConst ) found ).intValue();
                    fv.value = value;
                    str.value = str.value + fullname( found ) + " ";
                }
                else
                    if ( found.kind() == IdlType.e_enum_member )
                    {
                        value = ( ( IdlEnumMember ) found ).getValue();
                        fv.value = value;
                        str.value = str.value + fullname( found ) + "@ ";
                    }
                    else
                    {
                        parser.show_error( "This identifier " + name + " doesn't correspond to a constant" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                        parser.stopAt( parser.StopList );
                    }
            }

            break;

        case Token.t_par_ouverte :
            parser.symbole();
            str.value = str.value + "( ";
            value = const_exp( str, obj, fv );

            if ( parser.ctx.symb != Token.t_par_fermee )
            {
                parser.show_error( "')' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            str.value = str.value + ") ";
            parser.symbole();
            break;

        case Token.t_chaine :
            str.value = str.value + "\"" + parser.ctx.value + "\" ";
            parser.symbole();
            break;

        case Token.t_caractere :
            str.value = str.value + "'" + parser.ctx.value + "' ";
            parser.symbole();
            break;

        case Token.t_integer :
            str.value = str.value + parser.ctx.value;

            if ( parser.ctx.base == 1 )
                convertToDecimal();

            Long i = new Long( parser.ctx.value );

            value = i.longValue();

            fv.value = value;

            parser.symbole();

            break;

        case Token.t_real :
            str.value = str.value + parser.ctx.value + " ";

            Double d = new Double( parser.ctx.value );

            fv.value = d.doubleValue();

            parser.symbole();

            break;

        case Token.t_true :
            str.value = str.value + "true ";

            value = 1;

            parser.symbole();

            break;

        case Token.t_false :
            str.value = str.value + "false ";

            parser.symbole();

            value = 0;

            break;

        default :
            parser.show_error( "Incorrect expression" );

            parser.StopList.removeAllElements();

            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );

            parser.stopAt( parser.StopList );

            break;
        }

        return value;
    }

    // -------------
    // < UNARY EXP >
    // -------------
    /**
     * (22)
     * Ananlyse une expression constante
     */
    public long unary_exp( StringContainer str, IdlObject obj, DoubleContainer fv )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_moins :
            parser.symbole();
            str.value = str.value + "- ";
            return primary_exp( str, obj, fv );

        case Token.t_plus :
            parser.symbole();
            str.value = str.value + "+ ";
            return primary_exp( str, obj, fv );

        case Token.t_div :
            parser.symbole();
            str.value = str.value + "/ ";
            return primary_exp( str, obj, fv );

        case Token.t_mod :
            parser.symbole();
            str.value = str.value + "% ";
            return primary_exp( str, obj, fv );

        case Token.t_mul :
            parser.symbole();
            str.value = str.value + "* ";
            return primary_exp( str, obj, fv );

        case Token.t_tilde :
            parser.symbole();
            str.value = str.value + "~ ";
            return primary_exp( str, obj, fv );

        case Token.t_and :
            parser.symbole();
            str.value = str.value + "& ";
            return primary_exp( str, obj, fv );

        case Token.t_or :
            parser.symbole();
            str.value = str.value + "| ";
            return primary_exp( str, obj, fv );

        default :
            return primary_exp( str, obj, fv );
        }
    }

    protected boolean isIntoInterface( IdlObject cst )
    {
        IdlObject upper = final_type( cst.upper() );

        if ( ( upper.kind() == IdlType.e_interface ) || ( upper.kind() == IdlType.e_value ) )
            return true;

        return false;
    }

    // -------------
    // < CONST EXP >
    // -------------
    /**
     * (14)
     * Analyse une expression constante
     */
    public long const_exp( StringContainer str, IdlObject obj, DoubleContainer fv )
    {
        long size = 0;
        String name;
        IdlObject found;
        boolean flag = true;
        DoubleContainer fv2 = new DoubleContainer();

        while ( flag == true )
        {
            switch ( parser.ctx.symb )
            {

            case Token.t_moins :
                size = size - unary_exp( str, obj, fv2 );
                fv.value = fv.value - fv2.value;
                break;

            case Token.t_plus :
                size = size + unary_exp( str, obj, fv2 );
                fv.value = fv.value + fv2.value;
                break;

            case Token.t_div :
                size = size / unary_exp( str, obj, fv2 );
                fv.value = fv.value / fv2.value;
                break;

            case Token.t_mod :
                size = size % unary_exp( str, obj, fv2 );
                break;

            case Token.t_mul :
                size = size * unary_exp( str, obj, fv2 );
                fv.value = fv.value * fv2.value;
                break;

            case Token.t_tilde :
                size = unary_exp( str, obj, fv );
                break;

            case Token.t_and :
                size = size & unary_exp( str, obj, fv2 );
                break;

            case Token.t_or :
                size = size | unary_exp( str, obj, fv2 );
                break;

            case Token.t_quatre_pts :

            case Token.t_ident :
                name = scoped_name();

                if ( obj.isVisible( name, false ) == false )
                {
                    parser.show_error( "Undefined identifier : " + name );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                }
                else
                {
                    found = obj.returnVisibleObject( name, false );

                    if ( found.kind() == IdlType.e_const )
                    {
                        size = ( ( IdlConst ) found ).intValue();
                        fv.value = ( ( ( IdlConst ) found ).floatValue() );

                        if ( !isIntoInterface( found ) )
                            str.value = str.value + fullname( found ) + ".value ";
                        else
                            str.value = str.value + fullname( found ) + " ";
                    }
                    else
                        if ( found.kind() == IdlType.e_enum_member )
                        {
                            size = ( ( IdlEnumMember ) found ).getValue();
                            fv.value = size;
                            str.value = str.value + fullname( found ) + "@ ";
                        }
                        else
                        {
                            parser.show_error( "This identifier " + name + " doesn't correspond to a constant value" );
                            parser.StopList.removeAllElements();
                            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                            parser.stopAt( parser.StopList );
                        }
                }

                break;

            case Token.t_par_ouverte :
                size = primary_exp( str, obj, fv );
                break;

            case Token.t_chaine :
                str.value = str.value + "\"" + parser.ctx.value + "\" ";
                parser.symbole();
                break;

            case Token.t_caractere :
                str.value = str.value + "'" + parser.ctx.value + "' ";
                parser.symbole();
                break;

            case Token.t_integer :
                str.value = str.value + parser.ctx.value;

                if ( parser.ctx.base == 1 )
                    convertToDecimal();

                size = Long.valueOf( parser.ctx.value ).longValue();

                fv.value = size;

                parser.symbole();

                break;

            case Token.t_real :
                str.value = str.value + parser.ctx.value + " ";

                fv.value = Double.valueOf( parser.ctx.value ).doubleValue();

                parser.symbole();

                break;

            case Token.t_true :
                str.value = str.value + "true ";

                size = 1;

                parser.symbole();

                break;

            case Token.t_false :
                str.value = str.value + "false ";

                size = 0;

                parser.symbole();

                break;

            default :
                flag = false;

                break;
            }
        }

        return size;

    }

    // ----------------------
    // < POSITIVE INT CONST >
    // ----------------------
    /**
     * (26)
     * Analyse une expression entiere constante
     */
    public int positive_int_const( IdlObject obj )
    {
        StringContainer s = new StringContainer();
        DoubleContainer f = new DoubleContainer();
        int value = ( int ) const_exp( s, obj, f );

        if ( value < 0 )
            parser.show_error( "Positive interger constant expected" );

        return value;
    }

    // ---------
    // < FIXED >
    // ---------
    /**
     * (82)
     * Analyse la definition d'un type fixed
     */
    public void fixed_dcl( IdlObject obj )
    {
        int digits;
        int scale;

        if ( parser.ctx.symb != Token.t_fixed )
            parser.show_error( "'fixed' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_inf )
            parser.show_error( "'<' expected after 'fixed'" );

        parser.symbole();

        digits = positive_int_const( obj );

        if ( parser.ctx.symb != Token.t_virgule )
            parser.show_error( "',' expected" );

        parser.symbole();

        scale = positive_int_const( obj );

        if ( parser.ctx.symb != Token.t_sup )
        {
            parser.show_error( "'>' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.StopList.addElement( new Integer( Token.t_sup ) );
            parser.stopAt( parser.StopList );
        }

        parser.symbole();

        obj.addIdlObject( new IdlFixed( digits, scale, obj ) );
    }


    // ---------------
    // < FIXED CONST >
    // ---------------
    /**
     * (82)
     * Analyse la definition d'un type fixed constant
     */
    public void fixed_const_dcl( IdlObject obj )
    {
        if ( parser.ctx.symb != Token.t_fixed )
            parser.show_error( "'fixed' expected" );

        parser.symbole();

        obj.addIdlObject( new IdlFixed( 0, 0, obj ) );
    }


    // -----------
    // < WSTRING >
    // -----------
    /**
     * (67)
     * Analyse la definition d'une chaine de caracteres larges
     */
    public void wstring_dcl( IdlObject obj )
    {
        int size = 0;

        if ( parser.ctx.symb != Token.t_wstring )
            parser.show_error( "'wstring' expected" );

        parser.symbole();

        if ( parser.ctx.symb == Token.t_inf )
        {
            parser.symbole();

            size = positive_int_const( obj );

            if ( parser.ctx.symb != Token.t_sup )
            {
                parser.show_error( "'>' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.StopList.addElement( new Integer( Token.t_sup ) );
                parser.stopAt( parser.StopList );
            }

            parser.symbole();

        }

        obj.addIdlObject( new IdlWString( size, obj ) );
    }

    // ----------
    // < STRING >
    // ----------
    /**
     * (66)
     * Analyse la definition d'une chaine de caracteres
     */
    public void string_dcl( IdlObject obj )
    {
        int size = 0;

        if ( parser.ctx.symb != Token.t_string )
            parser.show_error( "'string' expected" );

        parser.symbole();

        if ( parser.ctx.symb == Token.t_inf )
        {
            parser.symbole();

            size = positive_int_const( obj );

            if ( parser.ctx.symb != Token.t_sup )
            {
                parser.show_error( "'>' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.StopList.addElement( new Integer( Token.t_sup ) );
                parser.stopAt( parser.StopList );
            }

            parser.symbole();

        }

        obj.addIdlObject( new IdlString( size, obj ) );
    }

    // ------------
    // < SEQUENCE >
    // ------------
    /**
     * (65)
     * Analyse la definition d'une sequence
     */
    public void sequence_dcl( IdlObject obj )
    {
        IdlSequence seq_obj;
        int size = 0;

        seq_obj = new IdlSequence( obj );

        seq_obj.attach_comment();

        if ( parser.ctx.symb != Token.t_sequence )
            parser.show_error( "'sequence' expected" );

        parser.symbole();

        if ( parser.ctx.symb == Token.t_inf )
        {
            parser.symbole();

            simple_type_spec( seq_obj );

            if ( parser.ctx.symb == Token.t_virgule )
            {
                parser.symbole();
                size = positive_int_const( obj );
            }

            seq_obj.setSize( size );

            if ( parser.ctx.symb != Token.t_sup )
            {
                parser.show_error( "'>' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            obj.addIdlObject( seq_obj );

        }

        obj.addIdlObject( new IdlString( size, obj ) );
    }

    // --------------
    // < CONST TYPE >
    // --------------
    /**
     * (13)
     * Analyse un type de constante
     */
    public void const_type( IdlObject obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_octet :
            obj.addIdlObject( IdlSimple.octet_type );
            break;

        case Token.t_boolean :
            obj.addIdlObject( IdlSimple.boolean_type );
            break;

        case Token.t_float :
            obj.addIdlObject( IdlSimple.float_type );
            break;

        case Token.t_double :
            obj.addIdlObject( IdlSimple.double_type );
            break;

        case Token.t_longdouble :
            obj.addIdlObject( IdlSimple.longdouble_type );
            break;

        case Token.t_short :
            obj.addIdlObject( IdlSimple.short_type );
            break;

        case Token.t_ushort :
            obj.addIdlObject( IdlSimple.ushort_type );
            break;

        case Token.t_long :
            obj.addIdlObject( IdlSimple.long_type );
            break;

        case Token.t_ulong :
            obj.addIdlObject( IdlSimple.ulong_type );
            break;

        case Token.t_longlong :
            obj.addIdlObject( IdlSimple.longlong_type );
            break;

        case Token.t_ulonglong :
            obj.addIdlObject( IdlSimple.ulonglong_type );
            break;

        case Token.t_char :
            obj.addIdlObject( IdlSimple.char_type );
            break;

        case Token.t_wchar :
            obj.addIdlObject( IdlSimple.wchar_type );
            break;

        case Token.t_wstring :
            wstring_dcl( obj );
            return ;

        case Token.t_string :
            string_dcl( obj );
            return ;

        case Token.t_fixed :
            fixed_const_dcl( obj );
            return ;

        case Token.t_quatre_pts :

        case Token.t_ident :
            String name = scoped_name();

            if ( obj.isVisible( name, false ) == false )
                parser.show_error( "Undefined idenfitier : " + name );
            else
                obj.addIdlObject( new IdlIdent( name, obj, obj.returnVisibleObject( name, false ) ) );

            return ;
        }

        parser.symbole();
    }

    // ---------
    // < CONST >
    // ---------
    /**
     * (12)
     * Analyse la definition d'une constante
     * <const> ::= 'const' <const_type> <identifier> '=' <const_exp>
     */
    public void const_dcl( IdlObject obj )
    {
        IdlConst const_obj;
        StringContainer exp = new StringContainer( "" );
        long value;

        const_obj = new IdlConst( obj );

        const_obj.attach_comment();

        if ( parser.ctx.symb != Token.t_const )
            parser.show_error( "'const' expected" );

        parser.symbole();

        const_type( const_obj );

        if ( parser.ctx.symb != Token.t_ident )
            parser.show_error( "Identifier expected" );
        else
        {
            const_obj.name( parser.ctx.value );

            if ( obj.isVisible( parser.ctx.value, true ) == true )
                parser.show_error( "Identifier already used : " + parser.ctx.value );

            parser.symbole();
        }

        if ( parser.ctx.symb != Token.t_egal )
        {
            parser.show_error( "'=' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }
        else
        {
            parser.symbole();

            DoubleContainer fvalue = new DoubleContainer( 0 );

            value = const_exp( exp, obj, fvalue );
            const_obj.expression( exp.value );
            const_obj.intValue( value );
            const_obj.floatValue( fvalue.value );
            obj.addIdlObject( const_obj );
            parser.ctx.one = parser.ctx.symb;
        }

    }

    // --------------------
    // < CONSTR_TYPE_SPEC >
    // --------------------
    /**
     * (33)
     * Ananlyse un type de donnee construit
     */
    public void constr_type_spec( IdlObject obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_struct :
            struct_dcl( obj );
            break;

        case Token.t_union :
            union_dcl( obj );
            break;

        case Token.t_enum :
            enum_dcl( obj );
            break;
        }

        parser.symbole();
    }

    // --------------------
    // < SIMPLE TYPE SPEC >
    // --------------------
    /**
     * (30)
     * Analyse un type de donnee simple
     */
    public void simple_type_spec( IdlObject obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_float :
            obj.addIdlObject( IdlSimple.float_type );
            break;

        case Token.t_double :
            obj.addIdlObject( IdlSimple.double_type );
            break;

        case Token.t_longdouble :
            obj.addIdlObject( IdlSimple.longdouble_type );
            break;

        case Token.t_short :
            obj.addIdlObject( IdlSimple.short_type );
            break;

        case Token.t_ushort :
            obj.addIdlObject( IdlSimple.ushort_type );
            break;

        case Token.t_long :
            obj.addIdlObject( IdlSimple.long_type );
            break;

        case Token.t_ulong :
            obj.addIdlObject( IdlSimple.ulong_type );
            break;

        case Token.t_longlong :
            obj.addIdlObject( IdlSimple.longlong_type );
            break;

        case Token.t_ulonglong :
            obj.addIdlObject( IdlSimple.ulonglong_type );
            break;

        case Token.t_char :
            obj.addIdlObject( IdlSimple.char_type );
            break;

        case Token.t_wchar :
            obj.addIdlObject( IdlSimple.wchar_type );
            break;

        case Token.t_fixed :
            fixed_dcl( obj );
            return ;

        case Token.t_wstring :
            wstring_dcl( obj );
            return ;

        case Token.t_string :
            string_dcl( obj );
            return ;

        case Token.t_quatre_pts :

        case Token.t_ident :
            String name = scoped_name();

            if ( obj.isVisible( name, false ) == false )
                parser.show_error( "Undefined identifier : " + name );
            else
                obj.addIdlObject( new IdlIdent( name, obj, obj.returnVisibleObject( name, false ) ) );

            return ;

        case Token.t_boolean :
            obj.addIdlObject( IdlSimple.boolean_type );

            break;

        case Token.t_octet :
            obj.addIdlObject( IdlSimple.octet_type );

            break;

        case Token.t_any :
            obj.addIdlObject( IdlSimple.any_type );

            break;

        case Token.t_object :
            obj.addIdlObject( IdlSimple.object_type );

            break;

        case Token.t_sequence :
            sequence_dcl( obj );

            break;

        case Token.t_ValueBase :
            obj.addIdlObject( IdlSimple.valuebase_type );

            break;
        }

        parser.symbole();
    }

    // -------------
    // < TYPE SPEC >
    // -------------
    /**
     * (29)
     * Ananlyse un type de donnee
     */
    public void type_spec( IdlObject obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_float :

        case Token.t_double :

        case Token.t_longdouble :

        case Token.t_short :

        case Token.t_ushort :

        case Token.t_long :

        case Token.t_ulong :

        case Token.t_longlong :

        case Token.t_ulonglong :

        case Token.t_char :

        case Token.t_wchar :

        case Token.t_boolean :

        case Token.t_octet :

        case Token.t_any :

        case Token.t_sequence :

        case Token.t_fixed :

        case Token.t_wstring :

        case Token.t_string :

        case Token.t_quatre_pts :

        case Token.t_ident :

        case Token.t_object :

        case Token.t_ValueBase :
            simple_type_spec( obj );
            break;

        case Token.t_struct :

        case Token.t_union :

        case Token.t_enum :
            constr_type_spec( obj );
            break;
        }
    }

    // ---------------
    // < DECLARATORS >
    // ---------------
    /**
     * (34)
     * Analyse une declaration de membre
     */
    public void declarators( IdlObject obj )
    {
        java.util.Vector dims;
        int dim;

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }
        else
        {
            if ( obj.upper().isVisible( parser.ctx.value, true ) == true )
                parser.show_error( "Identifier already used : " + parser.ctx.value );

            obj.name( parser.ctx.value );

            parser.symbole();

            if ( parser.ctx.symb == Token.t_cro_ouvert )
            {
                dims = new java.util.Vector();

                while ( true )
                {
                    parser.symbole();

                    dim = positive_int_const( obj );
                    dims.addElement( new Integer( dim ) );

                    if ( parser.ctx.symb != Token.t_cro_ferme )
                    {
                        parser.show_error( "']' expected" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                        parser.stopAt( parser.StopList );
                    }

                    parser.symbole();

                    if ( parser.ctx.symb != Token.t_cro_ouvert )
                        break;
                }


                IdlArray top_level = new IdlArray( obj );

                top_level.setDimension( ( ( Integer ) dims.elementAt( 0 ) ).intValue() );

                IdlArray array_obj = top_level;

                if ( dims.size() > 1 )
                {
                    for ( int i = 1; i < dims.size(); i++ )
                    {
                        IdlArray array_new = new IdlArray( array_obj );

                        array_new.setDimension( ( ( Integer ) dims.elementAt( i ) ).intValue() );

                        array_obj.addIdlObject( array_new );

                        array_obj = array_new;

                    }
                }

                array_obj.addIdlObject( obj.type() );
                obj.type( top_level );
            }
        }
    }

    // ----------
    // < MEMBER >
    // ----------
    /**
     * (56)
     * Analyse un membre d'une structure
     */
    public void member( IdlObject obj, java.util.Hashtable list )
    {
        IdlStructMember first_member_obj;
        IdlStructMember member_obj;

        first_member_obj = new IdlStructMember( obj );

        first_member_obj.attach_comment();

        type_spec( first_member_obj );
        first_member_obj.reset();

        while ( true )
        {
            member_obj = new IdlStructMember( obj );
            member_obj.type( first_member_obj.type() );
            member_obj.attach_comment( first_member_obj.getComment() );
            declarators( member_obj );

            switch ( member_obj.type().kind() )
            {

            case IdlType.e_array :

            case IdlType.e_sequence :
                parser.warning( "Anonymous sequences and arrays are deprecated" );
                break;

            case IdlType.e_string :

                if ( ( ( IdlString ) member_obj.type() ).max() != 0 )
                    parser.warning( "Anonymous unbounded string are deprecated" );

                break;

            case IdlType.e_wstring :
                if ( ( ( IdlWString ) member_obj.type() ).max() != 0 )
                    parser.warning( "Anonymous unbounded string are deprecated" );

                break;

            case IdlType.e_fixed :
                parser.warning( "Anonymous fixed are deprecated" );

                break;
            }

            obj.addIdlObject( member_obj );

            if ( list.get( member_obj.name() ) != null )
            {
                parser.show_error( "This identifier is already used : " + member_obj.name() );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }
            else
                list.put( member_obj.name(), member_obj );

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();
        }
    }

    // ---------------
    // < MEMBER LIST >
    // ---------------
    /**
     * (55)
     * Analyse les membres d'une structure
     */
    public void member_list( IdlObject obj )
    {
        java.util.Hashtable list = new java.util.Hashtable();

        while ( true )
        {
            member( obj, list );

            if ( parser.ctx.symb != Token.t_point_virgule )
            {
                parser.show_error( "';' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            //parser.symbole();
            block_symbole( obj );

            if ( parser.ctx.symb == Token.t_acc_fermee )
                break;
        }
    }

    // ----------
    // < STRUCT >
    // ----------
    /**
     * (54)
     * Analyse la definition d'une structure  
     */
    public void struct_dcl( IdlObject obj )
    {
        IdlStruct struct_obj;

        struct_obj = new IdlStruct( obj );
        struct_obj.attach_comment();

        org.openorb.compiler.parser.IdlParser.container = struct_obj;

        if ( parser.ctx.symb != Token.t_struct )
            parser.show_error( "'struct' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
            parser.show_error( "Identifier expected" );
        else
        {

            if ( obj.isVisible( parser.ctx.value, true ) )
            {
                // it may be a forward declaration

                IdlObject visible = obj.returnVisibleObject( parser.ctx.value, true );

                if ( visible.kind() == IdlType.e_struct )
                {
                    if ( ( ( IdlStruct ) visible ).isForward() )
                        ( ( IdlStruct ) visible ).setDefinition( struct_obj );
                    else
                    {
                        parser.show_error( "The '" + parser.ctx.value + "' structure is already defined..." );
                    }
                }
                else
                    parser.show_error( "Identifier already used : " + parser.ctx.value );
            }

            struct_obj.name( parser.ctx.value );
            obj.addIdlObject( struct_obj );
            parser.symbole();
        }

        if ( parser.ctx.symb != Token.t_acc_ouverte )
        {
            if ( parser.ctx.symb != Token.t_point_virgule )
            {
                parser.show_error( "';' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            struct_obj.isForward( true );
            parser.ctx.one = parser.ctx.symb;
            return ;
        }
        else
        {

            block_symbole( obj );
            member_list( struct_obj );

            if ( parser.ctx.symb != Token.t_acc_fermee )
            {
                parser.show_error( "'}' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }
        }

        org.openorb.compiler.parser.IdlParser.container = obj;
    }

    // --------------------
    // < SWITCH TYPE SPEC >
    // --------------------
    /**
    * (58)
    * Analyse un type de donnee de discriminant d'union
    */
    public void switch_type_spec( IdlObject obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_short :
            obj.addIdlObject( IdlSimple.short_type );
            break;

        case Token.t_ushort :
            obj.addIdlObject( IdlSimple.ushort_type );
            break;

        case Token.t_long :
            obj.addIdlObject( IdlSimple.long_type );
            break;

        case Token.t_ulong :
            obj.addIdlObject( IdlSimple.ulong_type );
            break;

        case Token.t_longlong :
            obj.addIdlObject( IdlSimple.longlong_type );
            break;

        case Token.t_ulonglong :
            obj.addIdlObject( IdlSimple.ulonglong_type );
            break;

        case Token.t_char :
            obj.addIdlObject( IdlSimple.char_type );
            break;

        case Token.t_wchar :
            obj.addIdlObject( IdlSimple.wchar_type );
            break;

        case Token.t_quatre_pts :

        case Token.t_ident :
            String name = scoped_name();

            if ( obj.isVisible( name, false ) == false )
                parser.show_error( "Undefined identifier : " + name );
            else
                obj.addIdlObject( new IdlIdent( name, obj, obj.returnVisibleObject( name, false ) ) );

            //obj.addIdlObject( obj.returnVisibleObject(name,false) );
            return ;

        case Token.t_boolean :
            obj.addIdlObject( IdlSimple.boolean_type );

            break;

        case Token.t_enum :
            enum_dcl( obj );

            break;

        default :
            parser.show_error( "Bad type for union discriminant" );

            break;
        }

        parser.symbole();
    }

    // ---------------
    // < SWITCH CASE >
    // ---------------
    /**
     * (60)
     * Analyse un des cas d'une union
     */
    public int switch_case( IdlObject obj, BooleanContainer asNext, java.util.Hashtable list )
    {
        int index = -1;
        StringContainer str = new StringContainer( "" );
        int size = 0;
        DoubleContainer fv = new DoubleContainer( 0 );

        asNext.value = false;

        switch ( parser.ctx.symb )
        {

        case Token.t_case :
            parser.symbole();
            size = ( int ) const_exp( str, obj, fv );
            break;

        case Token.t_default :
            parser.symbole();
            index = 0;
            break;

        default :
            parser.show_error( "Reserved words 'case' or 'default' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return index;
        }

        if ( parser.ctx.symb != Token.t_deux_points )
        {
            parser.show_error( "':' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }
        else
        {
            IdlUnionMember member_obj = new IdlUnionMember( obj );

            parser.symbole();

            if ( parser.ctx.symb != Token.t_case )
            {
                type_spec( member_obj );
                member_obj.reset();
                // member_obj.current()._upper = obj.upper(); // modify to supported nested items.

                declarators( member_obj );

                switch ( member_obj.type().kind() )
                {

                case IdlType.e_array :

                case IdlType.e_sequence :
                    parser.warning( "Anonymous sequences and arrays are deprecated" );
                    break;

                case IdlType.e_string :

                    if ( ( ( IdlString ) member_obj.type() ).max() != 0 )
                        parser.warning( "Anonymous unbounded string are deprecated" );

                    break;

                case IdlType.e_wstring :
                    if ( ( ( IdlWString ) member_obj.type() ).max() != 0 )
                        parser.warning( "Anonymous unbounded string are deprecated" );

                    break;

                case IdlType.e_fixed :
                    parser.warning( "Anonymous fixed are deprecated" );

                    break;
                }
            }
            else
            {
                member_obj.setAsNext();
                asNext.value = true;
            }

            if ( index == 0 )
                member_obj.setAsDefault();

            member_obj.setExpression( str.value );

            member_obj.setValue( size );

            if ( member_obj.name() != null )
            {
                if ( list.get( member_obj.name() ) != null )
                {
                    parser.show_error( "This identifier is already used : " + member_obj.name() );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                }
                else
                    list.put( member_obj.name(), member_obj );
            }

            obj.addIdlObject( member_obj );
        }

        return index;
    }

    // ---------------
    // < SWITCH BODY >
    // ---------------
    /**
     * (59)
     * Analyse le contenu d'une union
     */
    public void switch_body( IdlObject obj )
    {
        int idx = 0;
        int limit;
        boolean found = false;
        boolean stop;
        BooleanContainer asNext = new BooleanContainer();
        java.util.Hashtable list = new java.util.Hashtable();

        // Analyse la declaration de chaque element

        while ( true )
        {
            if ( switch_case( obj, asNext, list ) != -1 )
                ( ( IdlUnion ) obj ).index( idx );

            idx++;

            if ( asNext.value == false )
            {
                if ( parser.ctx.symb != Token.t_point_virgule )
                {
                    parser.show_error( "';' expected" );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                }

                //parser.symbole();
                block_symbole( obj );

                if ( parser.ctx.symb == Token.t_acc_fermee )
                    break;
            }
        }

        // Fixe le Type si un membre reference un autre membre

        int pos = obj.pos();

        obj.reset();

        while ( obj.end() != true )
        {
            if ( ( ( IdlUnionMember ) ( obj.current() ) ).isAsNext() )
            {
                int back = obj.pos();
                stop = false;
                found = false;

                while ( stop != true )
                {
                    if ( !( ( IdlUnionMember ) ( obj.current() ) ).isAsNext() )
                    {
                        found = true;
                        limit = obj.pos();
                        obj.current().reset();
                        IdlObject type = obj.current().current();
                        String name = obj.current().name();
                        obj.pos( back );

                        for ( int i = back; i < limit; i++ )
                        {
                            ( ( IdlUnionMember ) ( obj.current() ) ).memberTypeAndNameIs( type, name );
                            obj.next();
                        }

                        obj.pos( limit - 1 );
                        stop = true;
                    }

                    obj.next();

                    if ( obj.end() == true )
                        stop = true;
                }

                if ( found == false )
                    parser.show_error( "A type declaration is expected for union members" );
            }

            obj.next();
        }


        obj.pos( pos );
    }

    // ---------
    // < UNION >
    // ---------
    /**
     * (57)
     * Analyse la definition d'une union
     */
    public void union_dcl( IdlObject obj )
    {
        IdlUnion union_obj;

        union_obj = new IdlUnion( obj );

        union_obj.attach_comment();

        org.openorb.compiler.parser.IdlParser.container = union_obj;

        if ( parser.ctx.symb != Token.t_union )
            parser.show_error( "'union' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }
        else
        {

            if ( obj.isVisible( parser.ctx.value, true ) )
            {
                // it may be a forward declaration

                IdlObject visible = obj.returnVisibleObject( parser.ctx.value, true );

                if ( visible.kind() == IdlType.e_union )
                {
                    if ( ( ( IdlUnion ) visible ).isForward() )
                        ( ( IdlUnion ) visible ).setDefinition( union_obj );
                    else
                    {
                        parser.show_error( "The '" + parser.ctx.value + "' union is already defined..." );
                    }
                }
                else
                    parser.show_error( "Identifier already used : " + parser.ctx.value );
            }

            union_obj.name( parser.ctx.value );
            obj.addIdlObject( union_obj );
            parser.symbole();

            if ( parser.ctx.symb != Token.t_switch )
            {
                if ( parser.ctx.symb != Token.t_point_virgule )
                {
                    parser.show_error( "'switch' or ';' expected" );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                    return ;
                }

                union_obj.isForward( true );
                parser.ctx.one = parser.ctx.symb;
                return ;
            }
            else
            {
                parser.symbole();

                if ( parser.ctx.symb != Token.t_par_ouverte )
                {
                    parser.show_error( "'(' expected" );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                }
                else
                {
                    parser.symbole();

                    IdlUnionMember d = new IdlUnionMember( obj );
                    d.name( "__d" );
                    switch_type_spec( d );
                    union_obj.addIdlObject( d );

                    if ( parser.ctx.symb != Token.t_par_fermee )
                    {
                        parser.show_error( "')' expected" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                        parser.stopAt( parser.StopList );
                    }
                    else
                    {
                        parser.symbole();

                        if ( parser.ctx.symb != Token.t_acc_ouverte )
                        {
                            parser.show_error( "'{' expected" );
                            parser.StopList.removeAllElements();
                            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                            parser.stopAt( parser.StopList );
                        }
                        else
                        {
                            //parser.symbole();
                            block_symbole( obj );
                            switch_body( union_obj );

                            if ( parser.ctx.symb != Token.t_acc_fermee )
                            {
                                parser.show_error( "'}' expected" );
                                parser.StopList.removeAllElements();
                                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                                parser.stopAt( parser.StopList );
                            }

                        }
                    }
                }
            }
        }

        org.openorb.compiler.parser.IdlParser.container = obj;
    }

    // --------
    // < ENUM >
    // --------
    /**
     * (63)
     * Analyse une enumeration  
     */
    public void enum_dcl( IdlObject obj )
    {
        IdlEnum enum_obj;

        enum_obj = new IdlEnum( obj );

        enum_obj.attach_comment();

        org.openorb.compiler.parser.IdlParser.container = enum_obj;

        if ( parser.ctx.symb != Token.t_enum )
            parser.show_error( "'enum' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
            parser.show_error( "Identifier expected" );
        else
        {

            if ( obj.isVisible( parser.ctx.value, true ) )
                parser.show_error( "Identifier already used : " + parser.ctx.value );

            enum_obj.name( parser.ctx.value );

            parser.symbole();
        }

        obj.addIdlObject( enum_obj );

        if ( parser.ctx.symb != Token.t_acc_ouverte )
            parser.show_error( "'{ expected" );
        else
        {
            //parser.symbole();
            block_symbole( obj );
        }

        java.util.Hashtable list = new java.util.Hashtable();
        int index = 0;

        while ( true )
        {
            if ( parser.ctx.symb != Token.t_ident )
            {
                parser.show_error( "Identifier expected" );
                break;
            }
            else
            {
                IdlEnumMember member_obj = new IdlEnumMember( enum_obj );
                member_obj.name( parser.ctx.value );
                member_obj.setValue( index++ );

                if ( list.get( member_obj.name() ) != null )
                {
                    parser.show_error( "This identifier is already used : " + member_obj.name() );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                    return ;
                }
                else
                    list.put( member_obj.name(), member_obj );

                enum_obj.addIdlObject( member_obj );

                parser.symbole();

                if ( parser.ctx.symb != Token.t_virgule )
                    break;

                parser.symbole();
            }
        }

        if ( parser.ctx.symb != Token.t_acc_fermee )
            parser.show_error( "'}' expected" );

        org.openorb.compiler.parser.IdlParser.container = obj;
    }

    // -----------
    // < TYPEDEF >
    // -----------
    /**
     * (27)
     * Analyse la definition d'un nouveau type de donnee
     */
    public void type_dcl( IdlObject obj )
    {
        IdlTypeDef first_type_obj;
        IdlTypeDef type_obj;
        int count = 0;

        first_type_obj = new IdlTypeDef( obj );

        first_type_obj.attach_comment();

        if ( parser.ctx.symb != Token.t_typedef )
            parser.show_error( "'typedef' expected" );

        parser.symbole();

        type_spec( first_type_obj );

        // Add the sub type if it's a constructed type
        switch ( first_type_obj.type().kind() )
        {

        case IdlType.e_union :

        case IdlType.e_struct :

        case IdlType.e_enum :
            obj.addIdlObject( first_type_obj.type() );
            break;

        case IdlType.e_sequence :

            if ( ( ( IdlSequence ) first_type_obj.type() ).current().kind() == IdlType.e_sequence )
                parser.warning( "Anonymous sequences and arrays are deprecated" );

            break;
        }

        while ( true )
        {
            type_obj = new IdlTypeDef( obj );

            type_obj.type( first_type_obj.type() );

            type_obj.attach_comment( first_type_obj.getComment() );

            type_obj.setOrder( count++ );

            declarators( type_obj );

            obj.addIdlObject( type_obj );

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();
        }

        parser.ctx.one = parser.ctx.symb;
    }

    // -------------
    // < EXCEPTION >
    // -------------
    /**
     * (71)
     * Analyse la definition d'une exception
     */
    public void except_dcl( IdlObject obj )
    {
        IdlExcept except_obj;

        except_obj = new IdlExcept( obj );

        except_obj.attach_comment();

        if ( parser.ctx.symb != Token.t_exception )
            parser.show_error( "'exception' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
            parser.show_error( "Identifier expected" );
        else
        {

            if ( obj.isVisible( parser.ctx.value, true ) )
                parser.show_error( "Identifier already used : " + parser.ctx.value );

            except_obj.name( parser.ctx.value );

            parser.symbole();
        }

        if ( parser.ctx.symb != Token.t_acc_ouverte )
        {
            parser.show_error( "'{' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }
        else
        {
            parser.symbole();

            if ( parser.ctx.symb != Token.t_acc_fermee )
                member_list( except_obj );

            if ( parser.ctx.symb != Token.t_acc_fermee )
            {
                parser.show_error( "'}' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }
        }

        obj.addIdlObject( except_obj );

    }

    // ----------
    // < MODULE >
    // ----------
    /**
     * (3)
     * Analyse un module
     */
    public void module_dcl( IdlObject obj )
    {
        IdlModule module_obj;

        module_obj = new IdlModule( obj );

        module_obj.attach_comment();

        org.openorb.compiler.parser.IdlParser.container = module_obj;

        if ( parser.ctx.symb != Token.t_module )
            parser.show_error( "'module' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
            parser.show_error( "Identifier expected" );
        else
        {
            if ( obj.isVisible( parser.ctx.value, true ) )
            {
                module_obj = ( IdlModule ) obj.returnVisibleObject( parser.ctx.value, true );
                module_obj.refreshIncluded();
                org.openorb.compiler.parser.IdlParser.container = module_obj;

                if ( module_obj._import == false )
                {
                    parser.show_error( "This module ( " + parser.ctx.value + " ) is exposed but not imported, you cannot re-open it !" );
                    return ;
                }
            }
            else
            {
                module_obj.name( parser.ctx.value );
                obj.addIdlObject( module_obj );
            }

            parser.symbole();

        }

        if ( parser.ctx.symb != Token.t_acc_ouverte )
            parser.show_error( "'{' expected" );
        else
            parser.symbole();

        while ( true )
        {
            idl_definition( module_obj );

            if ( parser.ctx.symb == Token.t_acc_fermee )
                break;

            if ( parser.ctx.symb == Token.t_fin_fichier )
            {
                parser.show_error( "End of module expected for '" + module_obj.name() + "'" );
                return ;
            }
        }

        org.openorb.compiler.parser.IdlParser.container = obj;
    }

    /**
    * Returns the final definition of a data type
    *
    * @param obj the object
    * @return the final definition 
    */
    public IdlObject final_type( IdlObject obj )
    {
        switch ( obj.kind() )
        {

        case IdlType.e_ident :
            return final_type( ( ( IdlIdent ) obj ).internalObject() );

        case IdlType.e_typedef :

        case IdlType.e_union_member :

        case IdlType.e_struct_member :

        case IdlType.e_param :
            return final_type( obj.current() );

        default :
            return obj;
        }
    }

    // -------------------
    // < PARAM TYPE SPEC >
    // -------------------
    /**
     * (80)
     * Analyse le type d'un parametre
     */
    public void param_type_spec( IdlObject obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_float :
            obj.addIdlObject( IdlSimple.float_type );
            break;

        case Token.t_double :
            obj.addIdlObject( IdlSimple.double_type );
            break;

        case Token.t_longdouble :
            obj.addIdlObject( IdlSimple.longdouble_type );
            break;

        case Token.t_short :
            obj.addIdlObject( IdlSimple.short_type );
            break;

        case Token.t_ushort :
            obj.addIdlObject( IdlSimple.ushort_type );
            break;

        case Token.t_long :
            obj.addIdlObject( IdlSimple.long_type );
            break;

        case Token.t_ulong :
            obj.addIdlObject( IdlSimple.ulong_type );
            break;

        case Token.t_longlong :
            obj.addIdlObject( IdlSimple.longlong_type );
            break;

        case Token.t_ulonglong :
            obj.addIdlObject( IdlSimple.ulonglong_type );
            break;

        case Token.t_char :
            obj.addIdlObject( IdlSimple.char_type );
            break;

        case Token.t_wchar :
            obj.addIdlObject( IdlSimple.wchar_type );
            break;

        case Token.t_fixed :
            fixed_dcl( obj );
            return ;

        case Token.t_wstring :
            wstring_dcl( obj );
            return ;

        case Token.t_string :
            string_dcl( obj );
            return ;

        case Token.t_quatre_pts :

        case Token.t_ident :
            String name = scoped_name();

            if ( obj.isVisible( name, false ) == false )
                parser.show_error( "Undefined identifier " + name );
            else
            {
                // check the parameter type
                IdlObject ret = obj.returnVisibleObject( name, false );

                if ( final_type( ret ).kind() == IdlType.e_exception )
                {
                    parser.show_error( "Invalid type : " + ret.name() );
                }

                obj.addIdlObject( new IdlIdent( name, obj, ret ) );
            }

            return ;

        case Token.t_boolean :
            obj.addIdlObject( IdlSimple.boolean_type );
            break;

        case Token.t_octet :
            obj.addIdlObject( IdlSimple.octet_type );
            break;

        case Token.t_any :
            obj.addIdlObject( IdlSimple.any_type );
            break;

        case Token.t_object :
            obj.addIdlObject( IdlSimple.object_type );
            break;
            //-case Token.t_typecode :
            //- obj.addIdlObject( IdlSimple.typecode_type );
            //- break;

        case Token.t_ValueBase :
            obj.addIdlObject( IdlSimple.valuebase_type );
            break;
        }

        parser.symbole();
    }

    // --------
    // < ATTR >
    // --------
    /**
     * (70)
     * Analyse la definition d'un attribut
     */
    public void attr_dcl( IdlObject obj )
    {
        IdlAttribute first_attr_obj;
        IdlAttribute attr_obj;
        boolean rd = false;

        first_attr_obj = new IdlAttribute( obj );

        first_attr_obj.attach_comment();

        if ( parser.ctx.symb == Token.t_readonly )
        {
            parser.symbole();
            rd = true;
        }

        if ( parser.ctx.symb != Token.t_attribute )
        {
            parser.show_error( "'attribute' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();

        param_type_spec( first_attr_obj );

        switch ( first_attr_obj.type().kind() )
        {

        case IdlType.e_string :

            if ( ( ( IdlString ) first_attr_obj.type() ).max() != 0 )
                parser.warning( "Anonymous unbounded string are deprecated" );

            break;

        case IdlType.e_wstring :
            if ( ( ( IdlWString ) first_attr_obj.type() ).max() != 0 )
                parser.warning( "Anonymous unbounded string are deprecated" );

            break;

        case IdlType.e_fixed :
            parser.warning( "Anonymous fixed are deprecated" );

            break;
        }

        while ( true )
        {
            if ( parser.ctx.symb != Token.t_ident )
            {
                parser.show_error( "Identifier expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            if ( obj.isVisible( parser.ctx.value, true ) == true )
            {
                parser.show_error( "Identifier already used : " + parser.ctx.value );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            attr_obj = new IdlAttribute( obj );
            attr_obj.name( parser.ctx.value );
            attr_obj.readOnly( rd );
            attr_obj.type( first_attr_obj.type() );
            attr_obj.attach_comment( first_attr_obj.getComment() );

            obj.addIdlObject( attr_obj );

            parser.symbole();

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();
        }

        parser.ctx.one = parser.ctx.symb;
    }

    // ----------------
    // < OP TYPE SPEC >
    // ----------------
    /**
     * (74)
     * Analyse le type de retour d'une operation
     */
    public void op_type_spec( IdlOp obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_void :
            obj.addIdlObject( IdlSimple.void_type );
            parser.symbole();
            break;

        default :
            param_type_spec( obj );
            break;
        }
    }

    // -------------------
    // < PARAM ATTRIBUTE >
    // -------------------
    /**
     * (77)
     * Analyse l'attribut d'un parametre
     */
    public void param_attribute( IdlParam obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_in :
            obj.param_attr( 0 );
            break;

        case Token.t_out :
            obj.param_attr( 1 );
            break;

        case Token.t_inout :
            obj.param_attr( 2 );
            break;

        default :
            parser.show_error( "Attribute ( in, out, inout ) expected" );
            return ;
        }

        parser.symbole();
    }

    // -------------
    // < PARAM DCL >
    // -------------
    /**
     * (76)
     * Analyse la definition d'un parametre d'une operation
     */
    public void param_dcl( IdlOp obj )
    {
        IdlParam param_obj;

        param_obj = new IdlParam( obj );

        param_attribute( param_obj );
        param_type_spec( param_obj );

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected for parameter" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_par_fermee ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( obj.returnVisibleObject( parser.ctx.value, true ) != null )
        {
            parser.show_error( "Identifier already used" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_par_fermee ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        param_obj.name( parser.ctx.value );

        obj.addIdlObject( param_obj );

        parser.symbole();
    }

    // ------------------
    // < PARAMETER DCLS >
    // ------------------
    /**
     * (75)
     * Analyse les parametres d'une operation
     */
    public void parameter_dcls( IdlOp obj )
    {
        if ( parser.ctx.symb != Token.t_par_ouverte )
        {
            parser.show_error( "'(' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();

        while ( true )
        {
            if ( parser.ctx.symb == Token.t_par_fermee )
                break;

            param_dcl( obj );

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();

            if ( parser.ctx.symb == Token.t_par_fermee )
            {
                parser.show_error( "Parameter definition expected" );
                break;
            }

        }

        if ( parser.ctx.symb != Token.t_par_fermee )
        {
            parser.show_error( "')' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();
    }

    // ---------------
    // < RAISES EXPR >
    // ---------------
    /**
     * (78)
     * Analyse l'expression "raises" d'une operation
     */
    public void raises_expr( IdlObject obj )
    {
        java.util.Hashtable exceptions = new java.util.Hashtable();
        IdlRaises raises_obj;
        String name;

        raises_obj = new IdlRaises( obj );

        if ( parser.ctx.symb != Token.t_raises )
            parser.show_error( "'raises' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_par_ouverte )
        {
            parser.show_error( "'(' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_context ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();

        while ( true )
        {
            name = scoped_name();

            if ( obj.isVisible( name, false ) == false )
            {
                parser.show_error( "Undefined identifier : " + name );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_context ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            if ( obj.returnVisibleObject( name, false ).kind() != IdlType.e_exception )
            {
                parser.show_error( "This identifier doesn't correspond to an exception : " + name );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_context ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            IdlObject exp = obj.returnVisibleObject( name, false );

            raises_obj.addIdlObject( exp );

            if ( exceptions.get( exp ) != null )
            {
                parser.show_error( "An exception has been declared more than one time in the 'raise' clause : " + name );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_context ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }
            else
                exceptions.put( exp, exp );

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();
        }

        obj.addIdlObject( raises_obj );

        if ( parser.ctx.symb != Token.t_par_fermee )
        {
            parser.show_error( "')' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_context ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();
    }

    // ----------------
    // < CONTEXT EXPR >
    // ----------------
    /**
     * (79)
     * Analyse l'expression "context" d'une operation
     */
    public void context_expr( IdlOp obj )
    {
        IdlContext context_obj;
        String name;

        context_obj = new IdlContext( obj );

        if ( parser.ctx.symb != Token.t_context )
            parser.show_error( "'context' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_par_ouverte )
        {
            parser.show_error( "'(' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();

        while ( true )
        {

            if ( parser.ctx.symb != Token.t_chaine )
            {
                parser.show_error( "String expected after 'context'" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            context_obj.addValue( parser.ctx.value );
            parser.symbole();

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();
        }

        obj.addIdlObject( context_obj );

        if ( parser.ctx.symb != Token.t_par_fermee )
        {
            parser.show_error( "')' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_context ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();
    }

    // ------
    // < OP >
    // ------
    /**
     * (72)
     * Analyse la definition d'une operation
     */
    public void op_dcl( IdlObject obj )
    {
        IdlOp op_obj;
        boolean one = false;

        op_obj = new IdlOp( obj );

        op_obj.attach_comment();

        if ( parser.ctx.symb == Token.t_oneway )
        {
            parser.symbole();
            one = true;
        }

        op_obj.oneway( one );

        op_type_spec( op_obj );

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected for operation" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( obj.isVisible( parser.ctx.value, true ) )
        {
            parser.show_error( "Indentifier already used : " + parser.ctx.value );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_context ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        op_obj.name( parser.ctx.value );
        parser.symbole();

        parameter_dcls( op_obj );

        if ( parser.ctx.symb == Token.t_raises )
            raises_expr( op_obj );

        if ( parser.ctx.symb == Token.t_context )
            context_expr( op_obj );

        obj.addIdlObject( op_obj );

        parser.ctx.one = parser.ctx.symb;
    }

    // ----------
    // < EXPORT >
    // ----------
    /**
     * (9)
     * Analyse les membres d'une interface
     */
    public void export_dcl( IdlObject obj )
    {
        switch ( parser.ctx.symb )
        {

        case Token.t_const :
            const_dcl( obj );
            break;

        case Token.t_exception :
            except_dcl( obj );
            break;

        case Token.t_struct :
            struct_dcl( obj );
            break;

        case Token.t_union :
            union_dcl( obj );
            break;

        case Token.t_enum :
            enum_dcl( obj );
            break;

        case Token.t_typedef :
            type_dcl( obj );
            break;

        case Token.t_native :
            native_dcl( obj );
            break;

        case Token.t_pragma :
            pragma_dcl( obj );
            break;

        case Token.t_typeId :
            type_id_dcl( obj );
            break;

        case Token.t_typePrefix :
            type_prefix_dcl( obj );
            break;

        case Token.t_attribute :

        case Token.t_readonly :
            attr_dcl( obj );
            break;

        case Token.t_fixed :

        case Token.t_float :

        case Token.t_double :

        case Token.t_longdouble :

        case Token.t_short :

        case Token.t_ushort :

        case Token.t_long :

        case Token.t_ulong :

        case Token.t_longlong :

        case Token.t_ulonglong :

        case Token.t_char :

        case Token.t_wchar :

        case Token.t_wstring :

        case Token.t_string :

        case Token.t_quatre_pts :

        case Token.t_ident :

        case Token.t_any :

        case Token.t_void :
            //-case Token.t_typecode :

        case Token.t_object :

        case Token.t_boolean :

        case Token.t_octet :

        case Token.t_ValueBase :

        case Token.t_oneway :
            op_dcl( obj );
            break;

        default :
            parser.show_error( "Unexpected or undefined key word in the interface or value body of " + obj.name() );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            break;
        }

        parser.symbole();
    }

    // ------------------
    // < INTERFACE BODY >
    // ------------------
    /**
     * (8)
     * Analyse le corps d'une interface
     */
    public void interface_body( IdlInterface obj )
    {
        while ( true )
        {
            if ( parser.ctx.symb == Token.t_acc_fermee )
                break;

            export_dcl( obj );

            if ( parser.ctx.symb != Token.t_point_virgule )
            {
                parser.show_error( "';' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            parser.symbole();
        }
    }

    // --------------------
    // < INHERITANCE SPEC >
    // --------------------
    /**
     * (10)
     * Analyse les declarations d'heritages d'une interface
     */
    public void inheritance_spec( IdlInterface obj )
    {
        String name;
        IdlObject anObj;

        if ( parser.ctx.symb != Token.t_deux_points )
        {
            parser.show_error( "':' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        parser.symbole();

        while ( true )
        {
            name = scoped_name();

            if ( obj.isVisible( name, false ) == false )
            {
                parser.show_error( "Undeclared interface : " + name );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            anObj = obj.returnVisibleObject( name, false );

            if ( ( anObj.final_kind() != IdlType.e_forward_interface ) &&
                    ( anObj.final_kind() != IdlType.e_interface ) )
            {
                parser.show_error( "This identifier is not an interface : " + name );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                parser.stopAt( parser.StopList );
                break;
            }

            if ( anObj.final_kind() == IdlType.e_forward_interface )
            {
                if ( ( ( IdlInterface ) anObj.final_object() ).getInterface() == null )
                {
                    parser.show_error( "It is not legal to inherit from a forward interface which is not defined : " + name );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                    parser.stopAt( parser.StopList );
                    break;
                }
            }

            if ( obj.abstract_interface() )
            {
                if ( ( ( IdlInterface ) anObj.final_object() ).abstract_interface() == false )
                {
                    parser.show_error( "An abstract interface can only inherit from an abstract interface : " + name );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                    parser.stopAt( parser.StopList );
                    break;
                }
            }

            obj.addInheritance( anObj.final_object() );

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();
        }
    }

    // -----------------
    // < INTERFACE DCL >
    // -----------------
    /**
     * (5)
     * Analyse la declaration d'une interface
     */
    public void interface_dcl( IdlInterface obj )
    {
        if ( parser.ctx.symb == Token.t_deux_points )
            inheritance_spec( obj );

        if ( parser.ctx.symb != Token.t_acc_ouverte )
        {
            parser.show_error( "'{' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
            parser.stopAt( parser.StopList );
        }
        else
        {
            parser.symbole();
            interface_body( obj );
        }

        if ( parser.ctx.symb != Token.t_acc_fermee )
        {
            parser.show_error( "'}' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }
    }

    // -----------
    // < FORWARD >
    // -----------
    /**
     * (6)
     * Analyse une definition forward d'une interface
     */
    public void forward_dcl( IdlInterface obj )
    {
        obj.forward();
    }

    // -------------
    // < INTERFACE >
    // -------------
    /**
     * (4)
     * Analyse la definition d'une interface
     */
    public void interface_type( IdlObject obj )
    {
        IdlInterface interface_obj;
        IdlInterface anItf = null;
        IdlObject anObj;

        // Allready
        // 0 non declaree, non forwardee
        // 1 forwardee
        // 2 declaree
        int allready = 0;

        interface_obj = new IdlInterface( obj );

        interface_obj.attach_comment();

        org.openorb.compiler.parser.IdlParser.container = interface_obj;

        if ( parser.ctx.symb == Token.t_abstract )
        {
            interface_obj.abstract_interface( true );
            parser.symbole();
        }

        if ( parser.ctx.symb == Token.t_local )
        {
            if ( interface_obj.abstract_interface() )
                parser.show_error( "An abstract interface cannot be mark as 'local'" );

            interface_obj.local_interface( true );

            parser.symbole();
        }

        if ( parser.ctx.symb != Token.t_interface )
            parser.show_error( "'interface' expected" );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
            parser.show_error( "Identifier expected for interface" );
        else
        {
            interface_obj.name( parser.ctx.value );

            if ( obj.isVisible( parser.ctx.value, true ) == true )
            {
                anObj = obj.returnVisibleObject( parser.ctx.value, true );

                if ( ( anObj.kind() != IdlType.e_interface ) &&
                        ( anObj.kind() != IdlType.e_forward_interface ) )
                {
                    parser.show_error( "Identifier already used : " + parser.ctx.value );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_deux_points ) );
                    parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                    parser.stopAt( parser.StopList );
                }
                else
                {
                    anItf = ( IdlInterface ) anObj;

                    if ( anItf.isForward() == true )
                        allready = 1;
                    else
                        allready = 2;
                }
            }
        }

        obj.addIdlObject( interface_obj );

        parser.symbole();

        switch ( parser.ctx.symb )
        {

        case Token.t_point_virgule :

            if ( allready == 1 )
            {
                parser.show_error( "This interface is already declared : " + interface_obj.name() );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.stopAt( parser.StopList );
                return ;
            }

            forward_dcl( interface_obj );
            parser.ctx.one = parser.ctx.symb;
            org.openorb.compiler.parser.IdlParser.container = obj;
            break;

        case Token.t_acc_ouverte :

        case Token.t_deux_points :

            if ( allready == 1 )
            {
                if ( anItf.abstract_interface() != interface_obj.abstract_interface() )
                {
                    parser.show_error( "This interface is not defined as specified in forward declaraton : " + interface_obj.name() );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                    parser.stopAt( parser.StopList );
                    org.openorb.compiler.parser.IdlParser.container = obj;
                    return ;
                }
                else
                    interface_obj.defined( anItf );
            }

            if ( allready == 2 )
            {
                parser.show_error( "This interface is already declared : " + interface_obj.name() );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.stopAt( parser.StopList );
                org.openorb.compiler.parser.IdlParser.container = obj;
                return ;
            }

            interface_dcl( interface_obj );

            org.openorb.compiler.parser.IdlParser.container = obj;
            break;

        default :
            parser.show_error( "':' or  '{' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
            parser.stopAt( parser.StopList );
            break;
        }
    }

    // ----------
    // < NATIVE >
    // ----------
    /**
     * Analyse la definition d'un type natif
     */
    public void native_dcl( IdlObject obj )
    {
        IdlNative native_obj;

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected after 'native'" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( obj.isVisible( parser.ctx.value, false ) == true )
        {
            parser.show_error( "Identifier is already used : " + parser.ctx.value );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        native_obj = new IdlNative( obj );

        native_obj.attach_comment();

        native_obj.name( parser.ctx.value );

        obj.addIdlObject( native_obj );
    }

    // ---------------------
    // < VALUE FORWARD DCL >
    // ---------------------
    public void value_forward_dcl( IdlObject obj, String name, boolean abstract_value )
    {
        IdlValue value = new IdlValue( obj );

        if ( obj.isVisible( name, false ) )
        {
            parser.show_error( "Identifier already used" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }

        value.abstract_value( abstract_value );
        value.name( name );
        value.forward( true );

        obj.addIdlObject( value );

        parser.ctx.one = parser.ctx.symb;
        parser.ctx.symb = Token.t_point_virgule;
    }

    // -----------------
    // < VALUE BOX DCL >
    // -----------------
    public void value_box_dcl( IdlObject obj, String name )
    {

        if ( obj.isVisible( name, false ) )
        {
            parser.show_error( "Identifier already used" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }

        IdlValueBox value = new IdlValueBox( obj );

        value.attach_comment();

        value.name( name );

        type_spec( value );

        obj.addIdlObject( value );

        parser.ctx.one = parser.ctx.symb;
        parser.ctx.symb = Token.t_point_virgule;
    }

    // ---------------------
    // < VALUE INHERITANCE >
    // ---------------------
    public void value_inheritance( IdlObject obj, boolean custom, boolean statefull )
    {
        String vname = null;
        boolean trunc;
        boolean first = true;
        IdlObject inh = null;

        while ( true )
        {
            parser.symbole();

            IdlValueInheritance inheritance = new IdlValueInheritance( obj );

            if ( parser.ctx.symb == Token.t_truncatable )
            {
                if ( statefull == false )
                {
                    parser.show_error( "'truncatable' cannot be used for an abstract valuetype" );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                    parser.stopAt( parser.StopList );
                    return ;
                }

                if ( custom )
                {
                    parser.show_error( "'truncatable' cannot be used if a valuetype is specified with 'custom'" );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                    parser.stopAt( parser.StopList );
                    return ;
                }

                trunc = true;
                parser.symbole();
            }
            else
                trunc = false;

            vname = scoped_name();

            if ( obj.isVisible( vname, false ) )
            {
                IdlObject target = obj.returnVisibleObject( vname, false );
                inheritance.truncatable_member( trunc );

                if ( target.final_kind() == IdlType.e_forward_value )
                {
                    if ( ( ( IdlValue ) target ).definedValue() == null )
                    {
                        parser.show_error( "Unable to inherit from a forward valuetype which has not been yet defined" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                        parser.stopAt( parser.StopList );
                        return ;
                    }
                    else
                        inh = ( ( IdlValue ) target.final_object() ).definedValue();
                }
                else
                    inh = target.final_object();

                if ( inh.kind() != IdlType.e_value )
                {
                    parser.show_error( "Unable to inherit from a non-valuetype : " + vname );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                    parser.stopAt( parser.StopList );
                    return ;
                }

                if ( !custom )
                {
                    if ( ( ( IdlValue ) inh ).custom_value() )
                    {
                        parser.show_error( "A non-custom valuetype cannot inherit from a custom value type" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                        parser.stopAt( parser.StopList );
                        return ;
                    }
                }

                if ( ( first ) && ( trunc ) )
                {
                    if ( ( ( IdlValue ) inh ).abstract_value() )
                    {
                        parser.show_error( "An abstract valuetype cannot be specified with 'truncatable'" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                        parser.stopAt( parser.StopList );
                        return ;
                    }
                }

                if ( statefull == false )
                {
                    if ( ( ( IdlValue ) inh ).abstract_value() == false )
                    {
                        parser.show_error( "An abstract valuetype may only inherit from abstract valuetype" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                        parser.stopAt( parser.StopList );
                        return ;
                    }
                }

                if ( !first )
                {
                    if ( ( ( IdlValue ) inh ).abstract_value() == false )
                    {
                        parser.show_error( "The second and others inheritances must be abstract valuetype" );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_ouverte ) );
                        parser.stopAt( parser.StopList );
                        return ;
                    }
                }

                inheritance.addIdlObject( inh );

                ( ( IdlValue ) obj ).addInheritance( inheritance );

                if ( first )
                    first = false;
            }
            else
            {
                parser.show_error( "Undefined identifier : " + vname );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            //parser.symbole();

            if ( parser.ctx.symb != Token.t_virgule )
                break;
        }
    }

    // --------------------------
    // < VALUE INHERITANCE SPEC >
    // --------------------------
    public void value_inheritance_spec( IdlObject obj, boolean custom, boolean statefull )
    {
        java.util.Vector support_list = new java.util.Vector();


        if ( parser.ctx.symb == Token.t_deux_points )
            value_inheritance( obj, custom, statefull );

        if ( parser.ctx.symb == Token.t_supports )
        {
            while ( true )
            {
                parser.symbole();

                String supports_name = scoped_name();

                if ( obj.isVisible( supports_name , false ) )
                {
                    IdlObject target = obj.returnVisibleObject( supports_name, false );

                    if ( target.kind() == IdlType.e_forward_interface )
                    {
                        if ( ( ( IdlInterface ) target ).getInterface() == null )
                        {
                            parser.show_error( "Unable to inherit from an interface which has not been yet defined : " + parser.ctx.value );
                            parser.StopList.removeAllElements();
                            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                            parser.stopAt( parser.StopList );
                            return ;
                        }
                        else
                            target = ( ( IdlInterface ) target ).getInterface();
                    }

                    if ( target.kind() != IdlType.e_interface )
                    {
                        parser.show_error( "This identifier is not an interface : " + parser.ctx.value );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                        parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                        parser.stopAt( parser.StopList );
                        return ;
                    }
                    else
                        support_list.addElement( target );
                }
                else
                {
                    parser.show_error( "Undefined interface : " + parser.ctx.value );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                }

                if ( parser.ctx.symb != Token.t_virgule )
                    break;
            }

            int nb_non_abstract = 0;

            for ( int i = 0; i < support_list.size(); i++ )
            {
                if ( ( ( IdlInterface ) ( support_list.elementAt( i ) ) ).abstract_interface() == false )
                {
                    nb_non_abstract++;
                }
            }

            if ( nb_non_abstract > 1 )
            {
                parser.show_error( "A valuetype can only support one non abstract interface" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }


            ( ( IdlValue ) obj ).supports( support_list );

        }
    }

    // -----------------
    // < VALUE ABS DCL >
    // -----------------
    public void value_abs_dcl( IdlObject obj, String name )
    {
        IdlValue value = new IdlValue( obj );

        value.attach_comment();

        if ( obj.isVisible( name, false ) )
        {
            IdlObject target = obj.returnVisibleObject( name, false );

            if ( target.kind() == IdlType.e_value )
            {
                parser.show_error( "This valuetype is already defined : " + parser.ctx.value );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }
            else
                if ( target.kind() != IdlType.e_forward_value )
                {
                    parser.show_error( "This identifier is not a forward valuetype : " + parser.ctx.value );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                }
                else
                {
                    if ( ( ( IdlValue ) target ).abstract_value() == false )
                    {
                        parser.show_error( "The previous definition of this identifier was not for an abstract valuetype : " + parser.ctx.value );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                        parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                        parser.stopAt( parser.StopList );
                    }
                    else
                    {
                        if ( ( ( IdlValue ) target ).definedValue() != null )
                        {
                            parser.show_error( "Abstract valuetype already defined : " + parser.ctx.value );
                            parser.StopList.removeAllElements();
                            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                            parser.stopAt( parser.StopList );
                        }
                        else
                            ( ( IdlValue ) target ).definedValue( value );
                    }
                }
        }

        value.name( name );

        value.abstract_value( true );

        value_inheritance_spec( value, false, false );

        obj.addIdlObject( value );

        parser.symbole();

        while ( true )
        {
            if ( parser.ctx.symb == Token.t_acc_fermee )
                break;

            export_dcl( value );

            if ( parser.ctx.symb != Token.t_point_virgule )
            {
                parser.show_error( "';' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            parser.symbole();
        }

    }

    // --------------------
    // < STATE MEMBER DCL >
    // --------------------
    public void state_member_dcl( IdlObject obj )
    {
        IdlStateMember member = new IdlStateMember( obj );

        member.attach_comment();

        switch ( parser.ctx.symb )
        {

        case Token.t_public :
            member.public_member( true );
            break;

        case Token.t_private :
            member.public_member( false );
            break;
        }

        parser.symbole();

        type_spec( member );

        while ( true )
        {
            IdlStateMember member_obj = new IdlStateMember( obj );
            member_obj.public_member( member.public_member() );
            member_obj.type( member.type() );

            member_obj.attach_comment( member.getComment() );

            declarators( member_obj );

            obj.addIdlObject( member_obj );

            if ( parser.ctx.symb != Token.t_virgule )
                break;

            parser.symbole();
        }
    }

    // --------------------
    // < INIT PARAM DECLS >
    // --------------------
    public void init_param_decls( IdlObject obj )
    {
        while ( true )
        {
            parser.symbole();

            IdlFactoryMember member = new IdlFactoryMember( obj );

            if ( parser.ctx.symb != Token.t_in )
            {
                parser.show_error( "'in' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            parser.symbole();

            param_type_spec( member );

            if ( parser.ctx.symb != Token.t_ident )
            {
                parser.show_error( "Identifier expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            member.name( parser.ctx.value );

            obj.addIdlObject( member );

            parser.symbole();

            if ( parser.ctx.symb != Token.t_virgule )
                break;
        }
    }

    // ------------
    // < INIT DCL >
    // ------------
    public void init_dcl( IdlObject obj )
    {
        IdlFactory init = new IdlFactory( obj );

        init.attach_comment();

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected after 'factory'" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        init.name( parser.ctx.value );

        parser.symbole();

        init_param_decls( init );

        if ( parser.ctx.symb != Token.t_par_fermee )
        {
            parser.show_error( "')' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }

        parser.symbole();

        if ( parser.ctx.symb != Token.t_point_virgule )
        {
            parser.show_error( "';' expected" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
        }

        obj.addIdlObject( init );
    }

    // -------------
    // < VALUE DCL >
    // -------------
    public void value_dcl( IdlObject obj, String name, boolean custom )
    {

        IdlValue value = new IdlValue( obj );

        value.attach_comment();

        org.openorb.compiler.parser.IdlParser.container = value;

        if ( obj.isVisible( name, false ) )
        {
            IdlObject target = obj.returnVisibleObject( name, false );

            if ( target.kind() == IdlType.e_value )
            {
                parser.show_error( "This valuetype is already defined : " + parser.ctx.value );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }
            else
                if ( target.kind() != IdlType.e_forward_value )
                {
                    parser.show_error( "This identifier is not a forward valuetype : " + parser.ctx.value );
                    parser.StopList.removeAllElements();
                    parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                    parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                    parser.stopAt( parser.StopList );
                }
                else
                {
                    if ( ( ( IdlValue ) target ).abstract_value() )
                    {
                        parser.show_error( "The previous definition of this identifier was for an abstract valuetype : " + parser.ctx.value );
                        parser.StopList.removeAllElements();
                        parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                        parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                        parser.stopAt( parser.StopList );
                    }
                    else
                    {
                        if ( ( ( IdlValue ) target ).definedValue() != null )
                        {
                            parser.show_error( "Valuetype already defined : " + parser.ctx.value );
                            parser.StopList.removeAllElements();
                            parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                            parser.stopAt( parser.StopList );
                        }
                        else
                            ( ( IdlValue ) target ).definedValue( value );
                    }
                }
        }

        value.name( name );
        value.custom_value( custom );

        value_inheritance_spec( value, custom, true );

        obj.addIdlObject( value );

        parser.symbole();

        while ( true )
        {
            if ( parser.ctx.symb == Token.t_acc_fermee )
                break;

            switch ( parser.ctx.symb )
            {

            case Token.t_private :

            case Token.t_public :
                state_member_dcl( value );
                break;

            case Token.t_factory :
                init_dcl( value );
                break;

            default :
                export_dcl( value );
                break;
            }

            if ( parser.ctx.symb != Token.t_point_virgule )
            {
                parser.show_error( "';' expected" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_acc_fermee ) );
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
            }

            parser.symbole();
        }

        org.openorb.compiler.parser.IdlParser.container = obj;
    }

    // ------------------
    // < FULL VALUE DCL >
    // ------------------
    public void full_value_dcl( IdlObject obj )
    {
        boolean abstract_value = false;
        boolean custom = false;

        String value_name;

        if ( parser.ctx.symb == Token.t_abstract )
        {
            abstract_value = true;
            parser.symbole();
        }

        if ( parser.ctx.symb == Token.t_custom )
        {
            if ( abstract_value )
            {
                parser.show_error( "A 'abstract' value cannot be 'custom'" );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
                parser.stopAt( parser.StopList );
                return ;
            }
            else
            {
                custom = true;
                parser.symbole();
            }
        }

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected after 'valuetype'" );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        value_name = parser.ctx.value;

        parser.symbole();

        switch ( parser.ctx.symb )
        {

        case Token.t_point_virgule :
            value_forward_dcl( obj, value_name, abstract_value );
            return ;

        case Token.t_supports :

        case Token.t_deux_points :

        case Token.t_acc_ouverte :

            if ( abstract_value )
                value_abs_dcl( obj, value_name );
            else
                value_dcl( obj, value_name, custom );

            return ;

        default :
            value_box_dcl( obj, value_name );

            return ;
        }


    }

    // -------------
    // < CHANGE ID >
    // -------------
    /**
     * Change l'ID d'un objet CORBA
     */
    public void changeId( IdlObject current )
    {
        IdlObject obj;

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected after #pragma ID " );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }


        if ( current.isVisible( parser.ctx.value, false ) == false )
        {
            parser.show_error( "Undefined Identifier : " + parser.ctx.value );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        obj = current.returnVisibleObject( parser.ctx.value, false );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_chaine )
        {
            parser.show_error( "ID definition expected after #pragma " + obj.name() );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( obj.kind() == IdlType.e_forward_interface )
        {
            IdlInterface itf = ( IdlInterface ) obj;

            if ( itf.isForward() == true )
            {
                itf = itf.getInterface();

                if ( itf != null )
                    obj = itf;
            }
        }

        obj.setId( parser.ctx.value );
    }

    // ------------------
    // < CHANGE VERSION >
    // ------------------
    /**
     * Change la version d'un ID d'un objet CORBA
     */
    public void changeVersion( IdlObject current )
    {
        IdlObject obj;
        String id;
        String newId;

        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected after #pragma VERSION " );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }


        if ( current.isVisible( parser.ctx.value, false ) == false )
        {
            parser.show_error( "Undefined identifier : " + parser.ctx.value );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        obj = current.returnVisibleObject( parser.ctx.value, false );

        parser.symbole();

        if ( parser.ctx.symb != Token.t_real )
        {
            parser.show_error( "Bad version number for  : #pragma " + obj.name() );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        id = obj.getId();

        int index = id.lastIndexOf( ':' );

        if ( index != -1 )
            id = new String( id.substring( 0, index + 1 ) );
        else
            id = id + ":";

        newId = new String( id + parser.ctx.value );

        if ( obj.kind() == IdlType.e_forward_interface )
        {
            IdlInterface itf = ( IdlInterface ) obj;

            if ( itf.isForward() == true )
            {
                itf = itf.getInterface();

                if ( itf != null )
                {
                    obj.setId( newId );
                    obj = itf;
                }
            }
        }

        obj.setId( newId );
    }

    // -----------------
    // < CHANGE PREFIX >
    // -----------------
    /**
     * Change le prefixe des IDs des objets CORBA
     */
    public void changePrefix()
    {
        parser.symbole();

        if ( parser.ctx.symb != Token.t_chaine )
        {
            parser.show_error( "String expected after #pragma PREFIX " );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( parser.ctx.value.equals( "" ) )
            IdlObject.idlPrefix = null;
        else
            if ( checkPrefix( parser.ctx.value ) )
                IdlObject.idlPrefix = new String( parser.ctx.value );


    }

    // ----------------
    // < CHECK PREFIX >
    // ----------------

    /**
     * Check the pragma prefix string.
     * Any occurence of the ':' and the '/' character is illegal.
     *
     * From CORBA 2.4.2, 10.6.1
     * The second component is a list of identifiers, separated by '/' characters. These
     * identifiers are arbitrarily long sequences of alphabetic, digit, underscore ('_'),
     * hyphen ('-'), and period ('.') characters. Typically, the first identifier is a unique
     * prefix, and the rest are the OMG IDL Identifiers that make up the scoped name of
     * the definition.
     * Added by Olivier Modica, 4/11/2001
     */
    public boolean checkPrefix( java.lang.String pref )
    {
        for ( int i = 0; i < pref.length(); i++ )
        {

            char c = pref.charAt( i );

            if ( ( !java.lang.Character.isLetterOrDigit( c ) ) &&
                    ( c != '_' ) &&
                    ( c != '-' ) &&
                    ( c != '.' ) )
            {
                parser.show_error( "The character '" + c + "' is not allowed in a pragma prefix." );
                parser.StopList.removeAllElements();
                parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
                parser.stopAt( parser.StopList );
                return false;
            }

        }

        return true;
    }

    // ---------
    // < ECHO  >
    // ---------
    /**
     * Change le prefixe des IDs des objets CORBA
     */
    public void echo()
    {
        parser.symbole();

        if ( parser.ctx.symb != Token.t_chaine )
        {
            parser.show_error( "String expected after #pragma echo " );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        System.out.println( parser.ctx.value );
    }

    // ------------------
    // < CHANGE PACKAGE >
    // ------------------
    /**
     * Change le prefixe des IDs des objets CORBA
     */
    public void changePackage()
    {
        parser.symbole();

        if ( parser.ctx.symb != Token.t_chaine )
        {
            parser.show_error( "String expected after #pragma javaPackage " );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        org.openorb.compiler.IdlCompiler.use_package = true;
        org.openorb.compiler.IdlCompiler.packageName = new String( parser.ctx.value );
        //IdlObject.idlPrefix =
    }

    /**
     * Respond to #pragma javaNativeMap 
     */
    public void addNativeMap( IdlObject current )
    {
        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Identifier expected after #pragma javaNativeMap " );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( current.isVisible( parser.ctx.value, false ) == false )
        {
            parser.show_error( "Undefined identifier : " + parser.ctx.value );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        IdlObject obj = current.returnVisibleObject( parser.ctx.value, false );

        if ( ! ( obj instanceof IdlNative ) )
        {
            parser.show_error( "Attempted to apply native mapping to " + parser.ctx.value
                           + " which is not of native type." );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        String nativeMap = parser.ctx.value;

        parser.symbole();

        if ( parser.ctx.symb != Token.t_chaine )
        {
            parser.show_error( "String expected after #pragma javaNativeMap " + nativeMap );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        org.openorb.compiler.IdlCompiler.nativeDefinition.addElement( nativeMap + ":" + parser.ctx.value );
    }

    // --------------
    // < PRAGMA DCL >
    // --------------
    /**
     * Analyse les clauses pragma
     */
    public void pragma_dcl( IdlObject obj )
    {
        parser.symbole();

        if ( parser.ctx.symb != Token.t_ident )
        {
            parser.show_error( "Undefined pragma option : " + parser.ctx.value );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
        }
        else
        {
            if ( parser.ctx.value.toUpperCase().equals( "ID" ) == true )
                changeId( obj );
            else
                if ( parser.ctx.value.toUpperCase().equals( "VERSION" ) == true )
                    changeVersion( obj );
                else
                    if ( parser.ctx.value.toUpperCase().equals( "PREFIX" ) == true )
                        changePrefix();
                    else
                        if ( parser.ctx.value.toUpperCase().equals( "JAVAPACKAGE" ) == true )
                            changePackage();
                        else
                            if ( parser.ctx.value.toUpperCase().equals( "ECHO" ) == true )
                                echo();
                            else
                                if ( parser.ctx.value.toUpperCase().equals( "JAVANATIVEMAP" ) == true )
                                    addNativeMap( obj );
                                else
                                {
                                    parser.warning( "Unknown pragma directive : " + parser.ctx.value );
                                    parser.StopList.removeAllElements();
                                    parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
                                    parser.stopAt( parser.StopList );
                                }
        }

        parser.ctx.one = Token.t_point_virgule;
    }

    /**
     * Set a repository ID
     */
    public void type_id_dcl( IdlObject obj )
    {
        parser.symbole();

        String ident = scoped_name();

        if ( obj.isVisible( ident, false ) == false )
        {
            parser.show_error( "Unknown identifier : " + ident );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        IdlObject target = obj.returnVisibleObject( ident, false );

        if ( parser.ctx.symb != Token.t_chaine )
        {
            parser.show_error( "ID definition expected after typeId " + ident );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( target.kind() == IdlType.e_forward_interface )
        {
            IdlInterface itf = ( IdlInterface ) target;

            if ( itf.isForward() == true )
            {
                itf = itf.getInterface();

                if ( itf != null )
                    target = itf;
            }
        }

        target.setId( parser.ctx.value );
    }

    /**
     * Set repository IDs prefix
     */
    public void type_prefix_dcl( IdlObject obj )
    {
        parser.symbole();

        String ident = scoped_name();

        if ( obj.isVisible( ident, false ) == false )
        {
            parser.show_error( "Unknown identifier : " + ident );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_point_virgule ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        IdlObject target = obj.returnVisibleObject( ident, false );

        if ( parser.ctx.symb != Token.t_chaine )
        {
            parser.show_error( "ID definition expected after typeId " + ident );
            parser.StopList.removeAllElements();
            parser.StopList.addElement( new Integer( Token.t_fin_ligne ) );
            parser.stopAt( parser.StopList );
            return ;
        }

        if ( target.kind() == IdlType.e_forward_interface )
        {
            IdlInterface itf = ( IdlInterface ) target;

            if ( itf.isForward() == true )
            {
                itf = itf.getInterface();

                if ( itf != null )
                    target = itf;
            }
        }

        if ( target.equals( obj ) )
            IdlObject.idlPrefix = parser.ctx.value;

        target.applyPrefix( parser.ctx.value );
    }

    // --------------
    // < DEFINITION >
    // --------------
    /**
     * (1)
     * Analyse une definition IDL
     * <definition> ::= <type_dcl> ';' 
     *     |  <const_dcl> ';'
     *        | <except_dcl> ';'
     *     | <interface_dcl> ';'
     *     |  <module_dcl> ';'
     */
    public void idl_definition( IdlObject obj )
    {
        verbose( "<idl_definition> : " + parser.ctx.symb );

        switch ( parser.ctx.symb )
        {

        case Token.t_interface :
            interface_type( obj );
            break;

        case Token.t_module :
            module_dcl( obj );
            break;

        case Token.t_const :
            const_dcl( obj );
            break;

        case Token.t_exception :
            except_dcl( obj );
            break;

        case Token.t_struct :
            struct_dcl( obj );
            break;

        case Token.t_union :
            union_dcl( obj );
            break;

        case Token.t_enum :
            enum_dcl( obj );
            break;

        case Token.t_typedef :
            type_dcl( obj );
            break;

        case Token.t_pragma :
            pragma_dcl( obj );
            break;

        case Token.t_native :
            native_dcl( obj );
            break;

        case Token.t_local :
            parser.symbole();

            if ( parser.ctx.symb != Token.t_interface )
                parser.show_error( "'interface' key work expected after 'local'" );
            else
            {
                parser.ctx.one = parser.ctx.symb;
                parser.ctx.symb = Token.t_local;
                interface_type( obj );
            }

            break;

        case Token.t_abstract :
            parser.symbole();

            switch ( parser.ctx.symb )
            {

            case Token.t_valuetype :
                parser.ctx.one = parser.ctx.symb;
                parser.ctx.symb = Token.t_abstract;
                full_value_dcl( obj );
                break;

            case Token.t_interface :
                parser.ctx.one = parser.ctx.symb;
                parser.ctx.symb = Token.t_abstract;
                interface_type( obj );
                break;

            default :
                parser.show_error( "Bad definition after 'abstract'" );
                break;
            }

            break;

        case Token.t_custom :

        case Token.t_valuetype :
            full_value_dcl( obj );
            break;

        case Token.t_typeId :
            type_id_dcl( obj );
            break;

        case Token.t_typePrefix :
            type_prefix_dcl( obj );
            break;

        case Token.t_fin_fichier :
            return ;

        default :
            parser.show_error( "Definition expected" );
            break;
        }

        parser.symbole();

        if ( parser.ctx.symb != Token.t_point_virgule )
            parser.show_error( "';' expected" );

        parser.symbole();
    }

    // Next symbole for a block
    private void block_symbole( IdlObject obj )
    {
        verbose( "<block_symbole>" );

        parser.symbole();

        if ( parser.ctx.symb == Token.t_pragma )
        {
            pragma_dcl( obj );
            parser.symbole();
            block_symbole( obj );
        }
    }

    // ------------------
    // IMPORT DECLARATION
    // ------------------
    /**
     * Include external description
     */
    public void import_dcl( IdlObject obj )
    {
        verbose( "<import_dcl>" );

        String name = null;

        while ( true )
        {
            if ( parser.ctx.symb == Token.t_import )
            {
                parser.symbole();

                name = scoped_name();

                obj.addIdlObject( new IdlImport( obj, name ) );

                //irImport.getDescriptionFromIR( name, obj );

                if ( parser.ctx.symb != Token.t_point_virgule )
                    parser.show_error( "';' expected" );

                parser.symbole();
            }
            else
                break;
        }
    }

    // -----------------
    // IDL SPECIFICATION
    // -----------------
    /**
     * Premiere regle de la grammaire
     *
     * @param        obj   objet a partir duquel la compilation s'effectue
     */
    public void idl_specification( IdlObject obj )
    {
        verbose( "<idl_speficication>" );

        try
        {
            parser.symbole();

            while ( true )
            {
                if ( parser.ctx.symb == Token.t_import )
                    import_dcl( obj );

                idl_definition( obj );

                if ( parser.ctx.symb == Token.t_fin_fichier )
                    break;
            }
        }
        catch ( java.lang.Exception ex )
        {
            if ( org.openorb.compiler.IdlCompiler.verbose )
            {
                System.out.println( "--------------------" );
                ex.printStackTrace();
                System.out.println( "--------------------" );
                System.out.println( "" );
            }

            parser.show_internal_error();
        }
    }

    /**
     * Display a verbose message
     */
    private void verbose( String msg )
    {
        if ( org.openorb.compiler.IdlCompiler.verbose )
            System.out.println( msg );
    }
}
