/*
    This software may only be used by you under license from AT&T Corp.
    ("AT&T").  A copy of AT&T's Source Code Agreement is available at
    AT&T's Internet website having the URL:
    <http://www.research.att.com/sw/tools/graphviz/license/source.html>
    If you received this software without first entering into a license
    with AT&T, you have an infringing copy of this software and cannot use
    it without violating AT&T's intellectual property rights.
*/
#pragma prototyped

#include	<convert.h>
#ifdef HAVE_LIBEXPAT
#include	<expat.h>

#ifndef XML_STATUS_ERROR
#define XML_STATUS_ERROR 0
#endif

#define BUFSIZE		20000
#define SMALLBUF	1000
#define NAMEBUF		100

#define GXLATTR		"_gxl_"

#define TAG_NODE	1
#define TAG_EDGE	2
#define	TAG_GRAPH	3

#define	NODE_ATTR	1
#define	EDGE_ATTR	2
#define GLOB_ATTR	3

struct namemap_t {
	char name[SMALLBUF][NAMEBUF];
	char unique_name[SMALLBUF][NAMEBUF];
	int position;
};

typedef struct namemap_t *NameMap;

typedef struct userdata {
	char xml_attr_name[NAMEBUF];
	char xml_attr_value[SMALLBUF];
	char element_name[NAMEBUF][NAMEBUF];
	char composite_buffer[SMALLBUF];
	int element_count;
	int xmlAttrCharCount;
	int listen;
	int closedElementType;
	int globalAttrType;
	int compositeReadState;
	int edgeinverted;
	NameMap nameMap;
} userdata_t;

typedef userdata_t *UserData;

static int			Current_class, GSP;
static Agraph_t     *G, *root;
static Agraph_t		*Gstack[32];
static Agnode_t		*N;
static Agedge_t		*E;
static char			In_decl;


static UserData userdata_malloc() {	
	UserData user = (userdata_t*)malloc(sizeof(userdata_t));
	user->nameMap = (NameMap)malloc(sizeof(struct namemap_t));
	return user;
}

static void userdata_free(UserData ud) {
	free(ud->nameMap);
	free(ud);
}

static void init_udata(UserData ud) {
	ud->listen = FALSE;
	ud->element_count = 0;
	ud->globalAttrType = 0;
	ud->closedElementType = 0;
	ud->globalAttrType = 0;
	ud->compositeReadState = FALSE;
	ud->nameMap->position = 0;
	ud->edgeinverted = FALSE;
}

static void
addToMap(NameMap nm,char *gxlName, char *nodeName) {
	strcpy(nm->name[nm->position], gxlName);
	strcpy(nm->unique_name[nm->position++], nodeName);
}

static char
*mapLookup(NameMap nm, char *gxlName) {
	int i;
	for(i = 0; i < nm->position; i++) {
		if(strcmp(gxlName, nm->name[i]) == 0)
			return nm->unique_name[i];
	}
	return "";
}


static int 
isAnonGraph(char *name) {
	int i;
	if(*name == '%') {
		*name++;
		while(*name != '\0') {
			i = (int)*name++;
			if(i < 48 || i > 57) 
				return -1;
		}
		return 1;
	}
	return -1;
}


static void push_subg(Agraph_t *g)
{
	if(GSP == 0)
		root = g;
	G = Gstack[GSP++] = g;
}


static Agraph_t *pop_subg(void)
{
	Agraph_t		*g;
	if (GSP == 0) {
		fprintf(stderr,"Gstack underflow in graph parser\n"); exit(1);
	}
	g = Gstack[--GSP];					
	if (GSP > 0) G = Gstack[GSP - 1];	
	return g;
}


static Agnode_t *bind_node(const char *name)
{
	N = agnode(G,(char*)name,1);
	In_decl = FALSE;
	return N;
}


static Agedge_t *bind_edge(const char *tail, const char *head)
{
	Agnode_t *tailNode, *headNode;
    char*     key = 0;

	tailNode = agnode(G, (char*)tail,1);
	headNode = agnode(G, (char*)head,1);
	E = agedge(tailNode, headNode,key,1);
	In_decl = FALSE;
	return E;
}


static int get_xml_attr(char *attrname, const char **atts) {
	int count = 0;
	while(atts[count] != NULL) {
		if(strcmp(attrname, atts[count]) == 0) {
			return count + 1;
		}
		count += 2;
	}
	return -1;
}


static void attr_set(char *name, char *value, UserData ud)
{
	Agsym_t		*ap = NULL;
	char		*defval = "";
	char		portsBuf[SMALLBUF], gxlbuf[NAMEBUF];

	if (In_decl && (G->root == G)) defval = value;

	switch (Current_class) {
		case TAG_NODE:
			ap = agattr(root,AGNODE,name,0);
			if (ap == NULL) {
				ap = agattr(root,AGNODE,name,defval);
			}
			
			if(strcmp(name, "name") == 0) {
				char *oldName;
				char buf[NAMEBUF];
				int i;
				i = 0;
				oldName = agnameof(N);
				while(*oldName != '\0') {
					buf[i++] = *oldName++;
				}
				buf[i] = '\0';
				agrename((Agobj_t *)N, value);
				sprintf(gxlbuf, "%sid", GXLATTR);
				ap = agattr(root, AGNODE, gxlbuf, "");
				agxset(N,ap,buf);
				addToMap(ud->nameMap, buf, value);
			} else {			
				if(ud->globalAttrType == NODE_ATTR) {
					agattr(G, AGNODE, name, value);
				} else {
					agxset(N,ap,value);
				}
			}
			break;
		case TAG_EDGE:
			
			if(strcmp(name, "headport") == 0) {
				Agsym_t *attr;
				sprintf(portsBuf,":%s",value);

				if(ud->edgeinverted == TRUE) {
					ap = agattr(root,AGEDGE,"tailport",0);
					if (ap == NULL) {
						ap = agattr(root,AGEDGE,"tailport",defval);
					}
					if ((attr = agattr(G,AGEDGE,"tailport",NIL(char*))) == NILsym)
						attr = agattr(G,AGEDGE,"tailport","");
				} else {
					ap = agattr(root,AGEDGE,"headport",0);
					if (ap == NULL) {
						ap = agattr(root,AGEDGE,"headport",defval);
					}
					if ((attr = agattr(G,AGEDGE,"headport",NIL(char*))) == NILsym)
						attr = agattr(G,AGEDGE,"headport","");
				}
				agxset(E,attr,value);
			} else if(strcmp(name, "tailport") == 0) {
				Agsym_t *attr;
				sprintf(portsBuf,":%s",value);

				if(ud->edgeinverted == TRUE) {
					ap = agattr(root,AGEDGE,"headport",0);
					if (ap == NULL) {
						ap = agattr(root,AGEDGE,"headport",defval);
					}
					if ((attr = agattr(G,AGEDGE,"headport",NIL(char*))) == NILsym)
						attr = agattr(G,AGEDGE,"headport","");
				} else {
					ap = agattr(root,AGEDGE,"tailport",0);
					if (ap == NULL) {
						ap = agattr(root,AGEDGE,"tailport",defval);
					}
					if ((attr = agattr(G,AGEDGE,"tailport",NIL(char*))) == NILsym)
						attr = agattr(G,AGEDGE,"tailport","");
				}
				agxset(E,attr,value);
			} else {
				ap = agattr(root,AGEDGE,name,0);
				if (ap == NULL) {
					ap = agattr(root,AGEDGE,name,defval);
				}
				if(ud->globalAttrType == EDGE_ATTR) {
					agattr(G, AGEDGE, name, value);
				} else {
					agxset(E,ap,value);
				}
			}
			break; 
		case 0:		/* default */
		case TAG_GRAPH:
			/* GSP == 1; because the counter has already been incremented. */
			if(GSP == 1) {
				if(strcmp(name, "strict") == 0 && strcmp(value, "true") == 0) {
					G->desc.strict = 1;
					break;
				}
			}
			ap = agattr(root,AGRAPH,name,0);
			if (ap == NULL) {
				ap = agattr(root,AGRAPH,name,defval);
            } 

			if(strcmp(name, "name") == 0) {
				char *oldName;
				char buf[NAMEBUF];
				int i;
				i = 0;
				oldName = agnameof(G);
				while(*oldName != '\0') {
					buf[i++] = *oldName++;
				}
				buf[i] = '\0';
				agrename((Agobj_t *)G, value);
				sprintf(gxlbuf, "%sid", GXLATTR);
				ap = agattr(root, AGRAPH, gxlbuf, "");
				agxset(G,ap,buf);
				addToMap(ud->nameMap, buf, value);
			} else {			
	            agxset(G,ap,value);
			}
			break;
	}
}


/*------------- expat handlers ----------------------------------*/

static void
startElementHandler(void *userData, const char *name, const char **atts)
{
	int pos, i;
	char gxlbuf[NAMEBUF];
	UserData ud = (UserData)userData;
	Agraph_t *g = NULL;

	ud->xmlAttrCharCount = 0;

	if(strcmp(name, "gxl") == 0) {
		// do nothing
	} else if(strcmp(name, "graph") == 0) {
		const char *edgeMode="";
		const char *id;
		Current_class = TAG_GRAPH;

		if(ud->closedElementType == TAG_GRAPH) {
			fprintf(stderr, "Warning: Node contains more than one graph.\n");
		}
		id = atts[get_xml_attr("id", atts)];
		pos = get_xml_attr("edgemode", atts);
		if(pos > 0) {
			edgeMode = atts[pos];
		}

		if(GSP == 0) {
			if(strcmp(edgeMode, "directed") == 0) {
				g = agopen((char*)id,Agdirected,&AgDefaultDisc);			
			} else if(strcmp(edgeMode, "undirected") == 0) {
				g = agopen((char*)id,Agundirected,&AgDefaultDisc);			
			}
			push_subg(g);
			In_decl = TRUE;
		} else {
			if(isAnonGraph((char*)id) > 0) {
				static int		anon_id = 1;
				char			buf[SMALLBUF];
				char			c;
				Agraph_t			*subg;
				c = '%';

				In_decl = FALSE;
				sprintf(buf,"%c%d", c, anon_id++);
				subg = agsubg(G,buf,1);
				push_subg(subg);
				id = buf;
			} else {
				Agraph_t	 *subg;

				if (!(subg = agsubg(G,(char*)id,0))) 
				  subg = agsubg(G,(char*)id,1); 
				push_subg(subg);
				In_decl = FALSE;
			}
		}

		pos = get_xml_attr("role", atts);
		if(pos > 0) {
			sprintf(gxlbuf, "%srole", GXLATTR);
			attr_set(gxlbuf, (char*)atts[pos], ud);
		}

		pos = get_xml_attr("hypergraph", atts);
		if(pos > 0) {
			sprintf(gxlbuf, "%shypergraph", GXLATTR);
			attr_set(gxlbuf, (char*)atts[pos], ud);
		}

		i = 0;
		while(*id != '\0') {
			ud->element_name[ud->element_count][i++] = *id++;
		}
		ud->element_name[ud->element_count++][i] = '\0';
	} else if(strcmp(name, "node") == 0) {
		int i;
		i = 0;
		Current_class = TAG_NODE;		
		pos = get_xml_attr("id", atts);
		if(pos > 0) {
			const char *attrname;
			attrname = atts[pos];

			bind_node(attrname);		

			while(*attrname != '\0') {
				ud->element_name[ud->element_count][i++] = *attrname++;
			}
			ud->element_name[ud->element_count++][i] = '\0';
		}

	} else if(strcmp(name, "edge") == 0) {
		const char *head="", *tail="";
		char *tname;
		Agnode_t *t;

		Current_class = TAG_EDGE;
		pos = get_xml_attr("from", atts);
		if(pos > 0) 
			tail = atts[pos];
		pos = get_xml_attr("to", atts);
		if(pos > 0) 
			head = atts[pos];

		if(strcmp(mapLookup(ud->nameMap, (char*)tail), "") != 0)
			tail = mapLookup(ud->nameMap, (char*)tail);

		if(strcmp(mapLookup(ud->nameMap, (char*)head), "") != 0)
			head = mapLookup(ud->nameMap, (char*)head);

		bind_edge(tail, head);

		t = AGTAIL(E);
		tname = agnameof(t);

		if(strcmp(tname, tail) == 0) {
			ud->edgeinverted = FALSE;
		} else if(strcmp(tname, head) == 0){
			ud->edgeinverted = TRUE;
		}

		pos = get_xml_attr("fromorder", atts);
		if(pos > 0) {
			sprintf(gxlbuf, "%sfromorder", GXLATTR);
			attr_set(gxlbuf, (char*)atts[pos], ud);
		}

		pos = get_xml_attr("toorder", atts);
		if(pos > 0) {
			sprintf(gxlbuf, "%stoorder", GXLATTR);
			attr_set(gxlbuf, (char*)atts[pos], ud);
		}

		pos = get_xml_attr("id", atts);
		if(pos > 0) {
			sprintf(gxlbuf, "%sid", GXLATTR);
			attr_set(gxlbuf, (char*)atts[pos], ud);
		}
	} else if(strcmp(name, "attr") == 0) {
		int i;
		const char *attrname;
		i = 0;
		attrname = atts[get_xml_attr("name", atts)];

		while(*attrname != '\0') {
			ud->xml_attr_name[i++] = *attrname++;
		}
		ud->xml_attr_name[i] = '\0';
		pos = get_xml_attr("kind", atts);

		if(pos > 0) {
			if(strcmp("node", atts[pos]) == 0) 
				ud->globalAttrType = NODE_ATTR;
			else if(strcmp("edge", atts[pos]) == 0) 
				ud->globalAttrType = EDGE_ATTR;
			else if(strcmp("graph", atts[pos]) == 0) 
				ud->globalAttrType = GLOB_ATTR;
		} else {
			ud->globalAttrType = 0;
		}

	} else if(strcmp(name, "string") == 0
				|| strcmp(name, "bool") == 0
				|| strcmp(name, "int") == 0
				|| strcmp(name, "float") == 0) {

		ud->listen = TRUE;
		if(ud->compositeReadState) {
			sprintf(ud->composite_buffer, "%s<%s>", ud->composite_buffer, name);			
		}
	} else if(strcmp(name, "rel") == 0
				|| strcmp(name, "relend") == 0) {
		fprintf(stderr, "%s element is ignored by DOT\n", name);
	} else if(strcmp(name, "type") == 0) {
		pos = get_xml_attr("xlink:href", atts);
		if(pos > 0) {
			const char *href;
			char gxlbuf[NAMEBUF];
			href = atts[pos];

			sprintf(gxlbuf, "%stype", GXLATTR);
			attr_set(gxlbuf, (char*)href, ud);
		}		
	} else if(strcmp(name, "locator") == 0) {
		pos = get_xml_attr("xlink:href", atts);
		if(pos > 0) {
			int i;
			const char *href;
			char gxlbuf[NAMEBUF];
			href = atts[pos];

			sprintf(gxlbuf, "%slocator_%s", GXLATTR, href);
			for(i=0; gxlbuf[i] != '\0'; i++) {
				ud->xml_attr_value[i] = gxlbuf[i];				
			}
			ud->xmlAttrCharCount = i;
		}				
	} else if(strcmp(name, "seq") == 0
				|| strcmp(name, "set") == 0
				|| strcmp(name, "bag") == 0
				|| strcmp(name, "tup") == 0 
				|| strcmp(name, "enum") == 0 ) {
			
		if(ud->compositeReadState) {
			sprintf(ud->composite_buffer, "%s<%s>", ud->composite_buffer, name);			
		} else {
			ud->compositeReadState = TRUE;		
			sprintf(ud->composite_buffer, "<%s>", name);			
		}		
	} else {
		// must be some extension
		fprintf(stderr, "Unknown node %s; DOT does not support extensions.\n", name);
	}
}


static void
endElementHandler(void *userData, const char *name)
{
	UserData ud = (UserData)userData;
	char *ele_name;

	if(strcmp(name, "graph") == 0) {
		pop_subg();
		ele_name = ud->element_name[--ud->element_count];
		ud->closedElementType = TAG_GRAPH;
	} else if(strcmp(name, "node") == 0) {
		ele_name = ud->element_name[--ud->element_count];
		if(ud->closedElementType == TAG_GRAPH) {
			Agnode_t *node;
			node = agnode(root, ele_name, 0);
			agdelete(root, node);
		}
		Current_class = 0;
		ud->closedElementType = TAG_NODE;
	} else if(strcmp(name, "edge") == 0) {
		Current_class = 0;
		ud->closedElementType = TAG_EDGE;
		ud->edgeinverted = FALSE;
	} else if(strcmp(name, "attr") == 0) {
		ud->closedElementType = 0;

		switch(ud->globalAttrType) {
			case 0:
				break;
			case NODE_ATTR:
				Current_class = TAG_NODE;
				break;
			case EDGE_ATTR:
				Current_class = TAG_EDGE;
				break;
			case GLOB_ATTR:
			default:
				Current_class = TAG_GRAPH;
				break;
		}

		if(ud->compositeReadState) {
			char compAttrName[NAMEBUF];
			ud->compositeReadState = FALSE;
			sprintf(compAttrName, "%scomposite_%s", GXLATTR, ud->xml_attr_name);
			attr_set(compAttrName, ud->composite_buffer, ud);
			ud->xmlAttrCharCount = 0;
			ud->xml_attr_value[0] = '\0';
		} else {
			ud->xml_attr_value[ud->xmlAttrCharCount] = '\0';
			ud->xmlAttrCharCount = 0;
			attr_set(ud->xml_attr_name, ud->xml_attr_value, ud);
			ud->xml_attr_value[0] = '\0';
		}
	} else if(strcmp(name, "string") == 0
				|| strcmp(name, "bool") == 0
				|| strcmp(name, "int") == 0
				|| strcmp(name, "float") == 0) {
		ud->listen = FALSE;
		if(ud->compositeReadState) {
			sprintf(ud->composite_buffer, "%s</%s>", ud->composite_buffer, name);			
		}
	} else if(strcmp(name, "seq") == 0
				|| strcmp(name, "set") == 0
				|| strcmp(name, "bag") == 0
				|| strcmp(name, "tup") == 0 
				|| strcmp(name, "enum") == 0 ) {
			
		sprintf(ud->composite_buffer, "%s</%s>", ud->composite_buffer, name);			
	}
}


static void
characterDataHandler(void *userData, const char *s, int length) {
	UserData ud = (UserData)userData;
	int i, j;
	j = 0;
	
	if(!ud->listen) 
		return;

	if(ud->compositeReadState) {
		char substring[NAMEBUF];
		for(i=0; i < length; i++) {
			substring[i] = s[i];
		}
		substring[i] = '\0';
		sprintf(ud->composite_buffer, "%s%s", ud->composite_buffer, substring);					
		return;
	}

	for(i=0; i < length; i++) {
		ud->xml_attr_value[ud->xmlAttrCharCount + i] = s[i];
		j++;
	}
	ud->xmlAttrCharCount += j;
}


Agraph_t* 
gxl_to_dot(FILE* gxlFile) 
{
	char buf[BUFSIZE];
	int done;
	UserData udata = userdata_malloc();
	XML_Parser parser = XML_ParserCreate(NULL);

	init_udata(udata);
	XML_SetUserData(parser, udata);
	XML_SetElementHandler(parser, startElementHandler, endElementHandler);
	XML_SetCharacterDataHandler(parser, characterDataHandler);

    root = 0;
	do {
		size_t len = fread(buf, 1, sizeof(buf), gxlFile);
		if (len == 0) break;
		done = len < sizeof(buf);
		if (XML_Parse(parser, buf, len, done) == XML_STATUS_ERROR) {
			fprintf(stderr,
		          "%s at line %d\n",
		          XML_ErrorString(XML_GetErrorCode(parser)),
		          XML_GetCurrentLineNumber(parser));	
			exit(1);
	    }
	  } while (!done);
	XML_ParserFree(parser);
	userdata_free(udata);	
	
	return root;
}

#endif
