
/* This file is part of xtmodcoll-3.0, (c) xtmod.com
 * 
 * xtmodcoll-3.0 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * xtmodcoll-3.0 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser Public License
 * along with xtmodcoll-3.0.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xtmod.util.collections;

import java.util.ArrayList;
import java.util.List;

import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;


/**
 * This is an extension of {@link TreeDiff}, additionally capable of
 * merging incoming changes into the "existing" model. For this to work, the
 * "existing" model must consist of mutable tree nodes {@link MutableTreeNode}.
 */

@SuppressWarnings("unchecked")
public abstract class TreeMerge <C extends Comparable> extends TreeDiff<C> {

	/**
	 * Method to modify the existing model: Updates an existing node
	 * using data from the corresponding incoming node.
	 */
	abstract protected void applyUpdate (TreeNode exist, TreeNode incom);
	
	/**
	 * Method to modify the existing model: Creates a node for insertion
	 * into the existing model.
	 * <p>
	 * Return a shallow copy here, not a deep copy: invoking {@code clone()}
	 * should be just fine if your TreeNode class supports it.
	 *
	 * Regarding the user object, you are free to reuse the old one
	 * or create a new one.
	 */
	abstract protected MutableTreeNode create (TreeNode x);



	/** The exist model which we will write to. */
	protected DefaultTreeModel exist;

	/**
	 * Stores the exist model so it can be written to.
	 */
	@Override
	public void diff (DefaultTreeModel exist, DefaultTreeModel incom) {
		this.exist = exist;
		super.diff(exist, incom);
	}

	/** 
	 * Writes the diff results into the "existing" model.
	 * <p>
	 * This modifies the "existing" model, and publishes the according
	 * tree model events. Be sure to invoke this in the Swing Thread. 
	 */
	public void merge() {
		
		for (KidListDiff act: listdiffs) {


			// Update comes first, while the indices of the list are still stable
			// ---------------------------------------------------------------
			List<TreeNode> updated = act.getUpdated();
			List<TreeNode> updates = act.getUpdates();
			int nUpdate = updated.size();
			if (nUpdate > 0) {

				for (int i=0; i<nUpdate; i++) {
					TreeNode exist = updated.get(i);
					TreeNode incom = updates.get(i);
					applyUpdate(exist, incom);
				}

				List<Integer> updatedIndices = act.getUpdatedIndices();
				exist.nodesChanged(act.parent, toArray(updatedIndices));
			}

 
			// Remove comes second, it affects the indices. It however runs
			// backwards so lower indices do not get affected
			// ---------------------------------------------------------------
			List<TreeNode> removed = act.getRemoved();
			int nRemove = removed.size();
			if (nRemove > 0) {
				/* parent must be mutable to perform removal of children */
				MutableTreeNode parent = (MutableTreeNode)act.parent;

				List<Integer> removedIndices = act.getRemovedIndices();
				for (int i = nRemove-1; i >=0 ; i--)
					parent.remove(removedIndices.get(i));

				exist.nodesWereRemoved(parent, toArray(removedIndices), removed.toArray());
			}



			// Add comes third, it uses the indices that the elements had in 
			// their original (the incoming) list for inserting elements
			// in the existing list, so we end up with the same order.
			// ---------------------------------------------------------------

			/* When a node to be added contains subnodes, they must also
			 * be added of course. on the other hand, there should only
			 * be an added-event for the node to be added but not for the
			 * subnodes. the solution is to deepcopy the node, so by adding
			 * it we are in fact adding its whole subtree structure in one go. */
			List<MutableTreeNode> added = new ArrayList<MutableTreeNode>();
			for (TreeNode n : act.getAdded())
				added.add( deepcopy (n) );

			int nAdd = added.size();
			if (nAdd > 0) {
				/* parent must be mutable to perform addition of children */
				MutableTreeNode parent = (MutableTreeNode)act.parent;

				List<Integer> addedIndices = act.getAddedIndices();
				for (int i=0; i<nAdd; i++)
					parent.insert(added.get(i), addedIndices.get(i));

				exist.nodesWereInserted(parent, toArray(addedIndices));
			}

		}
	}

	protected int[] toArray (List<Integer> list) {
		int[] ret = new int[list.size()];
		for (int i=0; i<ret.length; i++)
			ret[i] = list.get(i);
		return ret;
	}
	
	protected MutableTreeNode deepcopy (TreeNode n) {
		MutableTreeNode copy = create(n);
		for (int i = 0; i < n.getChildCount(); i++) {
			TreeNode kid = n.getChildAt(i);
			MutableTreeNode kidCopy = deepcopy (kid);
			copy.insert (kidCopy, i);
		}
		return copy;
	}

	
}


