/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.tools.idlgen;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import org.openorb.compiler.IdlCompiler;
import org.openorb.compiler.object.IdlObject;
import org.openorb.compiler.parser.IdlParser;
import org.openorb.compiler.parser.IdlType;

import alma.tools.idlgen.comphelpgen.ComponentHelperGeneratorProxy;


/**
 * Main class of the ACS IDL compiler that produces interfaces, structs, and holders 
 * for use with XML binding classes.
 * The generated code will be similar but different from the output of a regular CORBA IDL compiler.  
 * IDL interfaces, structs, etc. that contain a field or parameter of type <code>XmlEntityStruct</code>
 * will be compiled into source code that contains a corresponding XML binding class. 
 * <p>
 * In collaboration with the other classes from this package, this class 
 * performs the following steps for each IDL file specified:
 * <ol>
 * <li>parses the IDL file (using an {@link org.openorb.compiler.parser.IdlParser}). The intermediate 
 * 		result is a tree of {@link org.openorb.compiler.object.IdlObject} nodes.</li>
 * <li>identifies IDL <code>typedef</code>s for the type <code>XmlEntityStruct</code>, 
 * 		e.g. the definition of the type <code>SchedBlock</code>.
 * 		(see {@link IdlTreeManipulator#findXmlTypedefNodes})</li>
 * <li>marks the nodes in the parse tree
 * 		that are affected by the XML entity types
 * 		(using {@link IdlTreeManipulator#findXmlEntityNodes}). 
 * 		This ensures that later we don't generate code that would be functionally identical 
 * 		to the code generated by a standard IDL compiler.<br>
 * 		Note that the identification of parse tree nodes works upward-recursive: for example, if an IDL
 * 		struct contains another struct which has a member of type <code>SchedBlock</code>, then
 * 		both IDL structs will get marked.</li>
 * <li>translates the relevant nodes of the manipulated IDL parse tree into Java code 
 * 		(using {@link JavaGenerator#translateData}). 
 * 		<br>
 * 		Note that the logic of mapping IDL to Java and the code generation itself are not separated.</li>
 * <li>optionally runs the code generator for component helper class templates,
 * 		see {@link alma.tools.idlgen.comphelpgen.ComponentHelperGeneratorProxy} and
 * 		{@link alma.acs.tools.comphelpergen.CompHelperGenerator}.<br>
 * 		Information about (component) interfaces and Java packages is transmitted as an XML string
 * 		that conforms to the schema <code>HelperInfo</code> in the package 
 * 		<code>alma.acs.tools.comphelpergen</code>.</li>
 * </ol> 
 * <p>
 * <b>Parameters and properties</b>:
 * <ul>
 * <li>Takes all command line arguments that are valid for {@link org.openorb.compiler.IdlCompiler},
 * 		although some of them don't apply and will be ignored.<br>
 * 		Interesting for tracing is <code>-verbose</code>.</li>
 * <li>The property <code>ACS.idl2jbind</code> specifies the mappings from typedef'd 
 * 		<code>XmlEntityStruct</code>s to Java binding classes.<br>
 * 		For example, the value  
 *      "<code>SchedBlock=alma.xmljbind.test.schedblock.SchedBlock;
 *      ObsProject=alma.xmljbind.test.obsproject.ObsProject</code>"
 * 		instructs the compiler to use the Java class 
 * 		<code>alma.xmljbind.test.schedblock.SchedBlock</code>
 * 		whereever an <code>XmlEntityStruct</code> typedef'd to <code>SchedBlock</code>
 * 		is found.</li>
 * <li>Two properties control the optional invocation of the code generator for 
 * 		component helper classes:
 * 		<ul>
 * 		<li><code>alma.acs.tools.comphelpergen.doGenerate</code>: if <code>true</code>, 
 * 			the ACS IDL compiler will run the generator after compiling an IDL file. 
 * 			See {@link alma.tools.idlgen.comphelpgen.ComponentHelperGeneratorProxy}.</li>
 * 		<li><code>alma.acs.tools.comphelpergen.outRootDir</code> specifies the directory under which
 * 			the component helper classes will be generated according to their Java packages.</li> 
 *		</ul></li>
 * </ul> 
 * In order to keep things simple, the implementation uses the OpenORB IDL compiler 
 * (<code>org.openorb.compiler.IdlCompiler</code> and related classes)
 * whereever this is possible. 
 * No OpenORB code is modified; this goal results in some less
 * intuitive handling of parse tree nodes, since new behavior could not be attached to the 
 * existing classes. On the other hand, we won't have to insert our patches into any new version 
 * of OpenORB that we might want to use in the future.
 * <p>  
 * A half-inspired attempt has been made to pave the road for code generation for languages other than Java. 
 * The tree manipulations could also be used by other code generators. 
 * For Java we reused OpenORB's implementation of the IDL2Java mapping logic 
 * (see {@link JavaGenerator}), but similar generators for other languages would be much more complex.
 * <p>
 * Remark on the proprietary code generation strategy used: 
 * Since OpenORB does not separate IDL2Java mapping logic from Java code generation, we can't
 * use any standard code generation framework without reimplementing the IDL2Java logic ourselves.
 * The choice was to accept being different from other ACS code generators, 
 * with the advantage of not writing lots of new code. 
 * 
 * @author hsommer
 */
public class XmlIdlCompiler
{
	
	public static final String PROP_DO_GENERATE_COMP_HELPERS = "alma.acs.tools.comphelpergen.doGenerate";
	public static final String PROP_COMP_HELPERS_OUTDIR = "alma.acs.tools.comphelpergen.outRootDir";
	
	protected static IdlCompiler s_idlCompiler = new IdlCompiler();
		

	/**
	 * Constructor for XmlIdlCompiler.
	 */
	public XmlIdlCompiler()
	{
		super();
	}

	/** 
	 * This operation is used to compile an IDL file
	 */
	public void compile_file(String file_name, IdlParser parser)
	{

		// -- options that are outside the OpenORB command line option handling
		
		String idl2jbindMapping = System.getProperty("ACS.idl2jbind");
		
		boolean doGenerateCompHelpers = Boolean.getBoolean(PROP_DO_GENERATE_COMP_HELPERS);
		String compHelperOutDir = System.getProperty(PROP_COMP_HELPERS_OUTDIR);

		if (IdlCompiler.verbose)
		{
			System.out.println("XmlIdl options (non-OpenORB):" +
								"\n  idl2jbindMapping = " + idl2jbindMapping + 
								"\n  doGenerateCompHelpers = " + doGenerateCompHelpers +
								"\n  compHelperOutDir = " + compHelperOutDir);		
		}
		
		if (doGenerateCompHelpers && compHelperOutDir == null)
		{
			System.err.println("can't generate component helper class(es) because " +
								"no output directory is given in alma.acs.tools.comphelpergen.outRootDir");
			doGenerateCompHelpers = false;
		}

		
		// -- create objects that do the work

		IdlTreeManipulator treeManipulator = new IdlTreeManipulator(JavaGenerator.getNamingConventions());
		
		ComponentHelperGeneratorProxy compHelpGen = null;
	
		if (doGenerateCompHelpers)
		{
			compHelpGen = new ComponentHelperGeneratorProxy(compHelperOutDir, IdlCompiler.verbose);
		}
		else if (IdlCompiler.verbose)
		{
			System.out.println("no code generation for component helper classes was selected...");
		}


		// -- build parse tree --
		
		IdlObject root = parser.compile_idl(file_name);

		if (IdlParser.totalError != 0)
		{
			String msg = "there are errors...compilation process stopped !";
			throw new RuntimeException(msg);
		}

		if (IdlCompiler.verbose)
		{		
			System.out.println("\n**** the original idl parse tree ****");
			treeManipulator.recursivePrint(root, 0);
			System.out.println("\n**** end of original idl parse tree ****\n");
		}

		IDLComponentTester.collectInterfaces(root);

		// -- init component helper generator with interfaces and Java packages --

		if (compHelpGen != null)	
		{
			try
			{
				compHelpGen.setOriginalParseTree(root, IdlCompiler.packageName);
			}
			catch (RuntimeException e)
			{
				System.err.println("failed to parse Java packages:");
				System.err.println(e.toString());
				System.err.println("Will disable component helper generation.");
				
				compHelpGen = null;
				doGenerateCompHelpers = false;
			}
//			treeManipulator.resetGraph(root);
		}
		
		
		// -- identify xml entity object structs --

		if (IdlCompiler.verbose)
		{
			System.out.println("\n**** detecting typedef'd xml entity structs ****");
		}
		Set xmlTypedefNodes = new LinkedHashSet();
		// do this separately from the previous compile_idl() pass, since some structs could be forward-declared
		treeManipulator.findXmlTypedefNodes(root, xmlTypedefNodes); 
		// print them 
		if (IdlCompiler.verbose)
		{
			for (Iterator iter = xmlTypedefNodes.iterator(); iter.hasNext();)
			{
				IdlObject node = (IdlObject) iter.next();
				System.out.println(IdlTreeManipulator.getTypeName(node.kind()) + "::" + node.getId());
			}
		}		
				
		// --  --

		Set nodesToBeGenerated = new LinkedHashSet();
		// unlike in OMG IDL2Java, we must generate code (holder class) for each xmlTypedefNode  
		nodesToBeGenerated.addAll(xmlTypedefNodes);
		if (IdlCompiler.verbose)
		{
			System.out.println("\n**** detecting usage of xml entity structs ****");
		}
		treeManipulator.findXmlEntityNodes(root, Collections.unmodifiableSet(xmlTypedefNodes), nodesToBeGenerated);
		treeManipulator.resetGraph(root);//to align ids with parent node names
		
		if (IdlCompiler.verbose)
		{
			System.out.println("\n**** nodes that need code generation ****");
			for (Iterator iter = nodesToBeGenerated.iterator(); iter.hasNext();)
			{
				IdlObject node = (IdlObject) iter.next();
				if (node.kind() != IdlType.e_ident)
				{
					System.out.println(IdlTreeManipulator.getTypeName(node.kind()) + "::" + node.name() + ": " + node.getId());
				}
			}
		
			System.out.println("\n**** the manipulated idl parse tree ****");
			treeManipulator.recursivePrint(root, 0);
		}
		
		
		// -- Start to generate Java code, the main purpose of all this --
		
		if (IdlCompiler.verbose)
		{
			System.out.println("\n**** Java code generation ****"); 
		}
		// otherwise we get an exception from the generator (translate_type(array))
		treeManipulator.resetGraph(root);
		
		JavaGenerator javaGen = new JavaGenerator(xmlTypedefNodes, nodesToBeGenerated);  
		javaGen.setIdlStruct2JavaBindingClassMappings(idl2jbindMapping);
		
		javaGen.translateData(root, IdlCompiler.packageName);
		
		if (IdlCompiler.verbose)
		{
			System.out.println("\n**** done with Java code generation ****"); 
		}


		// -- component helper code generation --

		if (compHelpGen != null)	
		{				
			System.out.println("ALMA IDL compiler done, now invoking component helper code generator...");
			compHelpGen.generateComponentHelperCode();
		}		
	}
	

	/**
	 * Modelled after {@link org.openorb.compiler.IdlCompiler#main}
	 * @param args 	same as for {@link IdlCompiler#scan_arguments(String[])}
	 */
	public static void main(String[] args)
	{
		if (args.length == 0)
			IdlCompiler.display_help();

		IdlParser parser = IdlCompiler.createIDLParser(args);

		if (IdlCompiler.idl_file_name.length == 0)
			IdlCompiler.display_help(); 

		XmlIdlCompiler idlCompiler = new XmlIdlCompiler();
		
		// -- Starts the compilation --
		try
		{
			java.util.Hashtable definedCopy =
				(java.util.Hashtable) IdlCompiler.definedMacros.clone();

			for (int i = 0; i < IdlCompiler.idl_file_name.length; i++)
			{
				IdlCompiler.definedMacros = definedCopy;

				System.out.println("ACS IDL compilation for " + IdlCompiler.idl_file_name[i]);
				idlCompiler.compile_file(IdlCompiler.idl_file_name[i], parser);
			}
		}
		catch (Exception ex)
		{
			System.err.println("ACS IDL compilation failed!");
			ex.printStackTrace(System.err);
			System.exit(1);
		}
	}
}
